"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectType = exports.InterfaceType = void 0;
const schema_field_1 = require("./schema-field");
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create an GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let schemaAddition = `interface ${this.name} {\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            const args = attribute.argsToString();
            schemaAddition = `${schemaAddition}  ${key}${args}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the field to add
     */
    addField(fieldName, field) {
        this.definition[fieldName] = field;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.directives = props.directives;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * Add a field to this Object Type
     *
     * @param fieldName - The name of the field
     * @param field - the resolvable field to add
     */
    addField(fieldName, field) {
        this.generateResolver(fieldName, field.fieldOptions);
        this.definition[fieldName] = field;
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        let title = this.name;
        if (this.interfaceTypes && this.interfaceTypes.length) {
            title = `${title} implements`;
            this.interfaceTypes.map((interfaceType) => {
                title = `${title} ${interfaceType.name},`;
            });
            title = title.slice(0, -1);
        }
        const directives = this.generateDirectives(this.directives);
        let schemaAddition = `type ${title} ${directives}{\n`;
        Object.keys(this.definition).forEach((key) => {
            const attribute = this.definition[key];
            const args = attribute.argsToString();
            schemaAddition = `${schemaAddition}  ${key}${args}: ${attribute.toString()}\n`;
        });
        return `${schemaAddition}}`;
    }
    /**
     * Utility function to generate directives
     *
     * @param directives the directives of a given type
     * @param delimiter the separator betweeen directives
     * @default - ' '
     */
    generateDirectives(directives, delimiter) {
        let schemaAddition = '';
        if (!directives) {
            return schemaAddition;
        }
        directives.map((directive) => {
            schemaAddition = `${schemaAddition}${directive.statement}${delimiter !== null && delimiter !== void 0 ? delimiter : ' '}`;
        });
        return schemaAddition;
    }
    /**
     * Generate the resolvers linked to this Object Type
     */
    generateResolver(fieldName, options) {
        if (options === null || options === void 0 ? void 0 : options.dataSource) {
            if (!this.resolvers) {
                this.resolvers = [];
            }
            this.resolvers.push(options.dataSource.createResolver({
                typeName: this.name,
                fieldName: fieldName,
                pipelineConfig: options.pipelineConfig,
                requestMappingTemplate: options.requestMappingTemplate,
                responseMappingTemplate: options.responseMappingTemplate,
            }));
        }
    }
}
exports.ObjectType = ObjectType;
//# sourceMappingURL=data:application/json;base64,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