"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualService = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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