"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
/**
 * Represents an identity source.
 *
 * The source can be specified either as a literal value (e.g: `Auth`) which
 * cannot be blank, or as an unresolved string token.
 */
class IdentitySource {
    /**
     * Provides a properly formatted header identity source.
     * @param headerName the name of the header the `IdentitySource` will represent.
     *
     * @returns a header identity source.
     */
    static header(headerName) {
        return IdentitySource.toString(headerName, 'method.request.header');
    }
    /**
     * Provides a properly formatted query string identity source.
     * @param queryString the name of the query string the `IdentitySource` will represent.
     *
     * @returns a query string identity source.
     */
    static queryString(queryString) {
        return IdentitySource.toString(queryString, 'method.request.querystring');
    }
    /**
     * Provides a properly formatted API Gateway stage variable identity source.
     * @param stageVariable the name of the stage variable the `IdentitySource` will represent.
     *
     * @returns an API Gateway stage variable identity source.
     */
    static stageVariable(stageVariable) {
        return IdentitySource.toString(stageVariable, 'stageVariables');
    }
    /**
     * Provides a properly formatted request context identity source.
     * @param context the name of the context variable the `IdentitySource` will represent.
     *
     * @returns a request context identity source.
     */
    static context(context) {
        return IdentitySource.toString(context, 'context');
    }
    static toString(source, type) {
        if (!source.trim()) {
            throw new Error('IdentitySources must be a non-empty string.');
        }
        return `${type}.${source}`;
    }
}
exports.IdentitySource = IdentitySource;
//# sourceMappingURL=data:application/json;base64,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