"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineDeployStackAction = void 0;
const cfn = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A class to deploy a stack that is part of a CDK App, using CodePipeline.
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 */
class PipelineDeployStackAction {
    constructor(props) {
        var _a, _b;
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxschema.ArtifactMetadataEntryType.ASSET);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: (_a = props.createChangeSetActionName) !== null && _a !== void 0 ? _a : 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: (_b = props.executeChangeSetActionName) !== null && _b !== void 0 ? _b : 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error('Cross-environment deployment is not supported');
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error('Use this action in a pipeline first before accessing \'deploymentRole\'');
        }
        return this._deploymentRole;
    }
    /**
     * Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,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