# -*- coding: utf-8 -*-
from collections import OrderedDict
from collective.taxonomy import LEGACY_PATH_SEPARATOR
from collective.taxonomy import NODE
from collective.taxonomy import PATH_SEPARATOR
from collective.taxonomy.i18n import _pmf
from collective.taxonomy.interfaces import ITaxonomy
from plone import api
from zope.component import queryMultiAdapter
from zope.component.hooks import getSite
from zope.i18nmessageid import MessageFactory
from zope.interface import implementer
from zope.schema.interfaces import IVocabularyFactory
from zope.schema.interfaces import IVocabularyTokenized
from zope.schema.vocabulary import SimpleTerm
from zope.schema.vocabulary import SimpleVocabulary
from zope.security.interfaces import IPermission


class Node(OrderedDict):
    """A non-persistent taxonomy tree node."""

    def __init__(self, term):
        self.term = term
        super(Node, self).__init__()


@implementer(IVocabularyFactory)
class TaxonomyVocabulary(object):
    # Vocabulary for generating a list of existing taxonomies

    def __call__(self, adapter):
        sm = getSite().getSiteManager()
        utilities = sm.getUtilitiesFor(ITaxonomy)
        return SimpleVocabulary(
            [
                SimpleTerm(value=utility.name, title=utility.title)
                for utility_name, utility in utilities
            ]
        )


@implementer(IVocabularyTokenized)
class Vocabulary(object):
    """Vocabulary, generated by the ITaxonomy utility"""

    def __init__(self, name, data, inv_data, order, version):
        self.data = data
        self.inv_data = inv_data
        self.order = order
        self.version = version
        self.message = MessageFactory(name)

    def __iter__(self):
        return iter(self.getTerms())

    def __len__(self):
        return len(self.getTerms())

    def __contains__(self, identifier):
        return self.getTerm(identifier) is not None

    def getTermByToken(self, input_identifier, tail_only=False):
        """Return term for token.

        If `tail_only` is provided, the title will be the last path
        element only.
        """

        if type(input_identifier) == list:
            raise LookupError("Expected string, not list")

        return SimpleTerm(
            value=input_identifier,
            title=self.message(
                input_identifier,
                self.inv_data[input_identifier],
                mapping={NODE: tail_only},
            ),
        )

    def getTermByValue(self, path):
        if self.version == 1:
            path = path.replace(PATH_SEPARATOR, LEGACY_PATH_SEPARATOR)
        return self.data[path]

    def getTerm(self, input_identifier):
        return self.getTermByToken(input_identifier)

    def getTerms(self, tail_only=False):
        """Return all terms.

        If `tail_only` is provided, the title will be the last path
        element only.
        """

        results = []

        if tail_only:
            mapping = {NODE: True}
        else:
            mapping = None

        for path, identifier in self.iterEntries():
            term = SimpleTerm(
                value=identifier, title=self.message(identifier, path, mapping=mapping)
            )
            results.append(term)

        return results

    def iterEntries(self):
        """Yields (path, identifier) pairs.

        Duplicate identifiers are omitted from output. Identifiers are
        returned in order (when ordering information is available).
        """

        identifiers = set()

        if self.version == 1:

            def fix(path):
                return path.replace(LEGACY_PATH_SEPARATOR, PATH_SEPARATOR)

        else:
            fix = lambda path: path  # noqa: E731

        if self.order is None:
            for path, identifier in self.data.items():
                if identifier in identifiers:
                    continue

                identifiers.add(identifier)
                yield fix(path), identifier
        else:
            for path in self.order.values():
                identifier = self.data.get(path)
                if identifier is None or identifier in identifiers:
                    continue

                identifiers.add(identifier)
                yield fix(path), identifier

    def makeTree(self):
        """Return term tree."""

        path_sep = (
            LEGACY_PATH_SEPARATOR if self.version == 1 else PATH_SEPARATOR
        )  # noqa: E501
        tree = OrderedDict()

        def add(path, value):
            parts = path.split(path_sep)[1:]
            node = tree
            for part in parts[:-1]:
                node = node[part]

            last = parts[-1]

            # The special flag NODE is provided in the translation
            # message mapping to indicate that the title should be
            # just the last term.
            title = self.message(value, path, mapping={NODE: True})
            term = SimpleTerm(value=value, title=title)
            node[last] = Node(term)

        for path, value in self.iterEntries():
            add(path, value)

        return tree


@implementer(IVocabularyFactory)
class PermissionsVocabulary(object):
    def __call__(self, context):
        result = []
        sm = getSite().getSiteManager()

        for (permission, permission_object) in sm.getUtilitiesFor(IPermission):
            result.append(
                SimpleTerm(
                    value=permission_object.id, title=_pmf(permission_object.title)
                )
            )

        result.sort(key=lambda permission: permission.title)
        return SimpleVocabulary(result)


@implementer(IVocabularyFactory)
class LanguagesVocabulary(object):
    """Languages vocabulary."""

    def __call__(self, context):
        portal = api.portal.get()
        terms = []
        portal_state = queryMultiAdapter(
            (portal, portal.REQUEST), name=u"plone_portal_state"
        )
        languages = portal_state.locale().displayNames.languages
        for token, value in sorted(languages.items()):
            terms.append(SimpleVocabulary.createTerm(token, token, value))

        return SimpleVocabulary(terms)
