# -*-coding: utf-8-*-

import pgpy
from pgpy.constants import PubKeyAlgorithm, KeyFlags, HashAlgorithm, SymmetricKeyAlgorithm, CompressionAlgorithm
import logging

def gen_pgp_key(name, email, comment="generated by sdata"):
    """generate a private key

    :param name:
    :param email:
    :return: key
    """

    # we can start by generating a primary key. For this example, we'll use RSA, but it could be DSA or ECDSA as well
    key = pgpy.PGPKey.new(PubKeyAlgorithm.RSAEncryptOrSign, 4096)

    # we now have some key material, but our new key doesn't have a user ID yet, and therefore is not yet usable!
    uid = pgpy.PGPUID.new(name, comment=comment, email=email)

    # now we must add the new user id to the key. We'll need to specify all of our preferences at this point
    # because PGPy doesn't have any built-in key preference defaults at this time
    # this example is similar to GnuPG 2.1.x defaults, with no expiration or preferred keyserver
    key.add_uid(uid, usage={KeyFlags.Sign, KeyFlags.EncryptCommunications, KeyFlags.EncryptStorage},
                hashes=[HashAlgorithm.SHA256, HashAlgorithm.SHA384, HashAlgorithm.SHA512, HashAlgorithm.SHA224],
                ciphers=[SymmetricKeyAlgorithm.AES256, SymmetricKeyAlgorithm.AES192, SymmetricKeyAlgorithm.AES128],
                compression=[CompressionAlgorithm.ZLIB, CompressionAlgorithm.BZ2, CompressionAlgorithm.ZIP,
                             CompressionAlgorithm.Uncompressed])
    return key

def get_key_from_blob(blob):
    """get key from blob

    :param blob:
    :return: key
    """
    keys = pgpy.PGPKey.from_blob(blob)
    logging.debug(keys)
    return keys[0]

def sign_table(data, sec_key, metadata_hash="!sdata_sha3_256_table", metadata_hash_signature="!sdata_sha3_256_table_signature"):
    """sign Data.table

    :param data:
    :param sec_key:
    :param metadata_hash: (default="!sdata_sha3_256_table")
    :param metadata_hash_signature: (default="!sdata_sha3_256_table_signature")
    :return:
    """
    sha3_256_table_signature = sec_key.sign(data.sha3_256_table)
    sha3_256_table_signature_str = str(sha3_256_table_signature)
    data.metadata.add(metadata_hash,
                      data.sha3_256_table)
    data.metadata.add(metadata_hash_signature,
                      sha3_256_table_signature_str,
                     dtype="str")

def verify_table(data, pub_key, metadata_hash="!sdata_sha3_256_table", metadata_hash_signature="!sdata_sha3_256_table_signature"):
    """verify Data.table

    :param data:
    :param pub_key:
    :param metadata_hash: (default="!sdata_sha3_256_table")
    :param metadata_hash_signature: (default="!sdata_sha3_256_table_signature")
    :return:
    """
    if not metadata_hash in data.metadata.keys() or not metadata_hash_signature in data.metadata.keys() :
        logging.error("data has no table signature")
        return False

    signature_verification = pub_key.verify(data.metadata[metadata_hash].value,
                          pgpy.PGPSignature.from_blob(data.metadata[metadata_hash_signature].value))
    if list(signature_verification.good_signatures)[0].verified is not True:
        logging.error("the table signature is False")
        return False

    if data.sha3_256_table == data.metadata[metadata_hash].value:
        return True
    else:
        return False
