"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionsProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
let SDK_METADATA = undefined;
/**
 * integ-tests can only depend on '@aws-cdk/core' so
 * this construct creates a lambda function provider using
 * only CfnResource
 */
class LambdaFunctionProvider extends constructs_1.Construct {
    constructor(scope, id /*, props?: LambdaFunctionProviderProps*/) {
        super(scope, id);
        this.policies = [];
        const staging = new aws_cdk_lib_1.AssetStaging(this, 'Staging', {
            sourcePath: path.join(__dirname, 'lambda-handler.bundle'),
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        const asset = stack.synthesizer.addFileAsset({
            fileName: staging.relativeStagedPath(stack),
            sourceHash: staging.assetHash,
            packaging: aws_cdk_lib_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const role = new aws_cdk_lib_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: [
                    {
                        PolicyName: 'Inline',
                        PolicyDocument: {
                            Version: '2012-10-17',
                            Statement: aws_cdk_lib_1.Lazy.list({ produce: () => this.policies }),
                        },
                    },
                ],
            },
        });
        const handler = new aws_cdk_lib_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Runtime: 'nodejs14.x',
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: aws_cdk_lib_1.Duration.minutes(2).toSeconds(),
                Handler: 'index.handler',
                Role: role.getAtt('Arn'),
            },
        });
        this.serviceToken = aws_cdk_lib_1.Token.asString(handler.getAtt('Arn'));
        this.roleArn = role.getAtt('Arn');
    }
    addPolicies(policies) {
        this.policies.push(...policies);
    }
}
/**
 * Mimic the singletonfunction construct in '@aws-cdk/aws-lambda'
 */
class SingletonFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.policies = [];
        this.lambdaFunction = this.ensureFunction(props);
        this.serviceToken = this.lambdaFunction.serviceToken;
        /**
         * The policies can be added by different constructs
         */
        this.node.addValidation({
            validate: () => {
                this.lambdaFunction.addPolicies(this.policies);
                return [];
            },
        });
    }
    ensureFunction(props) {
        const constructName = 'SingletonFunction' + slugify(props.uuid);
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        return new LambdaFunctionProvider(aws_cdk_lib_1.Stack.of(this), constructName);
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * singleton.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: 's3:GetObject',
     *   Resources: '*',
     * });
     */
    addToRolePolicy(statement) {
        this.policies.push(statement);
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        if (SDK_METADATA === undefined) {
            // eslint-disable-next-line
            SDK_METADATA = require('./sdk-api-metadata.json');
        }
        const srv = service.toLowerCase();
        const iamService = (SDK_METADATA[srv] && SDK_METADATA[srv].prefix) || srv;
        const iamAction = api.charAt(0).toUpperCase() + api.slice(1);
        this.policies.push({
            Action: [`${iamService}:${iamAction}`],
            Effect: 'Allow',
            Resource: resources || ['*'],
        });
    }
}
/**
 * Represents an assertions provider. The creates a singletone
 * Lambda Function that will create a single function per stack
 * that serves as the custom resource provider for the various
 * assertion providers
 */
class AssertionsProvider extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.policies = [];
        this.handler = new SingletonFunction(this, 'AssertionsProvider', {
            uuid: '1488541a-7b23-4664-81b6-9b4408076b81',
            policies: aws_cdk_lib_1.Lazy.list({ produce: () => this.policies }),
        });
        this.handlerRoleArn = this.handler.lambdaFunction.roleArn;
        this.serviceToken = this.handler.serviceToken;
    }
    /**
     * Encode an object so it can be passed
     * as custom resource parameters. Custom resources will convert
     * all input parameters to strings so we encode non-strings here
     * so we can then decode them correctly in the provider function
     */
    encode(obj) {
        if (!obj) {
            return obj;
        }
        return JSON.parse(JSON.stringify(obj), (_k, v) => {
            switch (v) {
                case true:
                    return 'TRUE:BOOLEAN';
                case false:
                    return 'FALSE:BOOLEAN';
                default:
                    return v;
            }
        });
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        this.handler.addPolicyStatementFromSdkCall(service, api, resources);
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * @example
     * declare const provider: AssertionsProvider;
     * provider.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: 's3:GetObject',
     *   Resources: '*',
     * });
     */
    addToRolePolicy(statement) {
        this.handler.addToRolePolicy(statement);
    }
}
exports.AssertionsProvider = AssertionsProvider;
_a = JSII_RTTI_SYMBOL_1;
AssertionsProvider[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AssertionsProvider", version: "2.39.1-alpha.0" };
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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