"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scope = {
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        };
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scope = {
            complianceResourceTypes: types,
        };
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scope = {
            tagKey: key,
            tagValue: value,
        };
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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