# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mailers',
 'mailers.encrypters',
 'mailers.preprocessors',
 'mailers.signers',
 'mailers.transports']

package_data = \
{'': ['*']}

install_requires = \
['anyio>=3.3.0,<4.0.0']

extras_require = \
{'dkim': ['dkimpy>=1.0,<2.0'],
 'jinja2': ['jinja2>=3.0,<4.0'],
 'smtp': ['aiosmtplib>=2.0,<3.0']}

entry_points = \
{'pytest11': ['mailers = mailers.pytest_plugin']}

setup_kwargs = {
    'name': 'mailers',
    'version': '3.0.1',
    'description': 'Email delivery for asyncio.',
    'long_description': '# Mailers for asyncio\n\n![PyPI](https://img.shields.io/pypi/v/mailers)\n![GitHub Workflow Status](https://img.shields.io/github/workflow/status/alex-oleshkevich/mailers/Lint%20and%20test)\n![GitHub](https://img.shields.io/github/license/alex-oleshkevich/mailers)\n![Libraries.io dependency status for latest release](https://img.shields.io/librariesio/release/pypi/mailers)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/mailers)\n![GitHub Release Date](https://img.shields.io/github/release-date/alex-oleshkevich/mailers)\n![Lines of code](https://img.shields.io/tokei/lines/github/alex-oleshkevich/mailers)\n\n## Features\n\n* fully typed\n* full utf-8 support\n* async and sync sending\n* pluggable transports\n* multiple built-in transports including: SMTP, file, null, in-memory, streaming, and console.\n* message preprocessors\n* embeddables\n* attachments (with async and sync interfaces)\n* message signing via Signer interface (DKIM bundled)\n* message encryption via Encrypter interface\n* trio support via anyio\n* fallback transports\n* global From address\n* templated emails\n\n## Usage\n\n```bash\npip install mailers[aiosmtplib]\n```\n\nThen create mailer:\n\n```python\nfrom mailers import Mailer\n\nmailer = Mailer("smtp://user:password@localhost:25?timeout=2")\nawait mailer.send_message(\n    to="user@localhost", from_address="from@localhost", subject="Hello", text="World!"\n)\n```\n\n## Compose messages\n\nIf you need more control over the message, you can use `Email` object to construct email message and then send it\nusing `mailer.send` method.\n\n```python\nfrom mailers import Email, Mailer\n\nmessage = Email(\n    to="user@localhost",\n    from_address="from@example.tld",\n    cc="cc@example.com",\n    bcc=["bcc@example.com"],\n    text="Hello world!",\n    html="<b>Hello world!</b>",\n)\nmailer = Mailer("smtp://")\nawait mailer.send(message)\n```\n\n### Global From address\n\nInstead of setting "From" header in every message, you can set it mailer-wide. Use `from_address` argument of Mailer\nclass:\n\n```python\nmailer = Mailer(from_address="sender@localhost")\n```\n\nThe mailer will set From header with the given value to all messages that do not container From or Sender headers.\n\n## Using Jinja templates\n\n> Requires `jinja2` package installed\n\nYou can use Jinja to render templates. This way, your `text` and `html` can be rendered from a template.\n\nUse `TemplatedMailer` instead of default `Mailer` and set a `jinja2.Environment` instance.\nThen, call `send_templated_message`.\n\n```python\nimport jinja2\n\nfrom mailers import TemplatedMailer\n\nenv = jinja2.Environment(loader=jinja2.FileSystemLoader(["templates"]))\nmailer = TemplatedMailer("smtp://", env)\nmailer.send_templated_message(\n    to="...",\n    subject="Hello",\n    text_template="mail.txt",\n    html_template="mail.html",\n    template_context={"user": "root"},\n)\n```\n\n## Attachments\n\nUse `attach`, `attach_from_path`, `attach_from_path_sync` methods to attach files.\n\n```python\nfrom mailers import Email\n\nmessage = Email(\n    to="user@localhost", from_address="from@example.tld", text="Hello world!"\n)\n\n# attachments can be added on demand\nawait message.attach_from_path("file.txt")\n\n# or use blocking sync version\nmessage.attach_from_path_sync("file.txt")\n\n# attach from variable\nmessage.attach("CONTENTS", "file.txt", "text/plain")\n```\n\n## Embedding files\n\nIn the same way as with attachments, you can inline file into your messages. This is commonly used to display embedded\nimages in the HTML body. Here are method you can use `embed`, `embed_from_path`, `embed_from_path_sync`.\n\n```python\nfrom mailers import Email\n\nmessage = Email(\n    to="user@localhost",\n    from_address="from@example.tld",\n    html=\'Render me <img src="cid:img1">\',\n)\n\nawait message.embed_from_path(path="/path/to/image.png", name="img1")\n```\n\nNote, that you have to add HTML part to embed files. Otherwise, they will be ignored.\n\n## Message signatures\n\nYou can sign messages (e.g. with DKIM) by passing `signer` argument to the `Mailer` instance.\n\n```python\nsigner = MySigner()\nmailer = Mailer(..., signer=signer)\n```\n\n### DKIM signing\n\n> Requires `dkimpy` package installed\n\nYou may wish to add DKIM signature to your messages to prevent them from being put into the spam folder.\n\nNote, you need to install [`dkimpy`](https://pypi.org/project/dkimpy/) package before using this feature.\n\n```python\nfrom mailers import Mailer\nfrom mailers.signers.dkim import DKIMSigner\n\nsigner = DKIMSigner(selector="default", private_key_path="/path/to/key.pem")\n\n# or you can put key content using private_key argument\nsigner = DKIMSigner(selector="default", private_key="PRIVATE KEY GOES here...")\n\nmailer = Mailer("smtp://", signer=signer)\n```\n\nNow all outgoing messages will be signed with DKIM method.\n\nThe plugin signs "From", "To", "Subject" headers by default. Use "headers" argument to override it.\n\n## Custom signers\n\nExtend `mailers.Signer` class and implement `sign` method:\n\n```python\nfrom email.message import Message\nfrom mailers import Signer\n\n\nclass MySigner(Signer):\n    def sign(self, message: Message) -> Message:\n        # message signing code here...\n        return message\n```\n\n## Encrypters\n\nWhen encrypting a message, the entire message (including attachments) is encrypted using a certificate. Therefore, only\nthe recipients that have the corresponding private key can read the original message contents.\n\n````python\nencrypter = MyEncrypter()\nmailer = Mailer(..., encrypter=encrypter)\n````\n\nNow all message content will be encrypted.\n\n## Custom encrypters\n\nExtend `mailers.Encrypter` class and implement `encrypt` method:\n\n```python\nfrom email.message import Message\nfrom mailers import Encrypter\n\n\nclass MyEncrypter(Encrypter):\n    def encrypt(self, message: Message) -> Message:\n        # message encrypting code here...\n        return message\n```\n\n## High Availability\n\nUse `MultiTransport` to provide a fallback transport. By default, the first transport is used but if it fails to send\nthe message, it will retry sending using next configured transport.\n\n```python\nfrom mailers import Mailer, MultiTransport, SMTPTransport\n\nprimary_transport = SMTPTransport()\nfallback_transport = SMTPTransport()\n\nmailer = Mailer(MultiTransport([primary_transport, fallback_transport]))\n```\n\n## Preprocessors\n\nPreprocessors are function that mailer calls before sending. Preprocessors are simple functions that modify message\ncontents.\n\nBelow you see an example preprocessor:\n\n```python\nfrom email.message import EmailMessage\n\nfrom mailers import Mailer\n\n\ndef attach_html_preprocessor(message: EmailMessage) -> EmailMessage:\n    message.add_alternative(b"This is HTML body", subtype="html", charset="utf-8")\n    return message\n\n\nmailer = Mailer(preprocessors=[attach_html_preprocessor])\n```\n\n### CSS inliner\n\n> Requires `toronado` package installed\n\nOut of the box we provide `mailers.preprocessors.css_inliner` utility that converts CSS classes into inline styles.\n\n## Transports\n\n### SMTP transport\n\n> Requires `aiosmtplib` package installed\n\nSend messages via third-party SMTP servers.\n\n**Class:** `mailers.transports.SMTPTransport`\n**directory** `smtp://user:pass@hostname:port?timeout=&use_tls=1`\n**Options:**\n\n* `host` (string, default "localhost") - SMTP server host\n* `port` (string, default "25") - SMTP server port\n* `user` (string) - SMTP server login\n* `password` (string) - SMTP server login password\n* `use_tls` (string, choices: "yes", "1", "on", "true") - use TLS\n* `timeout` (int) - connection timeout\n* `cert_file` (string) - path to certificate file\n* `key_file` (string) - path to key file\n\n### File transport\n\nWrite outgoing messages into a directory in EML format.\n\n**Class:** `mailers.transports.FileTransport`\n**DSN:** `file:///tmp/mails`\n**Options:**\n\n* `directory` (string) path to a directory\n\n### Null transport\n\nDiscards outgoing messages. Takes no action on send.\n\n**Class:** `mailers.transports.NullTransport`\n**DSN:** `null://`\n\n### Memory transport\n\nKeeps all outgoing messages in memory. Good for testing.\n\n**Class:** `mailers.transports.InMemoryTransport`\n**DSN:** `memory://`\n**Options:**\n\n* `storage` (list of strings) - outgoing message container\n\nYou can access the mailbox via ".mailbox" attribute.\n\nExample:\n\n```python\nfrom mailers import Mailer, InMemoryTransport, Email\n\ntransport = InMemoryTransport([])\nmailer = Mailer(transport)\n\nawait mailer.send(Email(...))\nassert len(transport.mailbox) == 1  # here are all outgoing messages\n```\n\n### Streaming transport\n\nWrites all messages into a writable stream. Ok for local development.\n\n**Class:** `mailers.transports.StreamTransport`\n**DSN:** unsupported\n**Options:**\n\n* `output` (typing.IO) - a writable stream\n\nExample:\n\n```python\nimport io\nfrom mailers import Mailer, StreamTransport\n\ntransport = StreamTransport(output=io.StringIO())\nmailer = Mailer(transport)\n```\n\n### Console transport\n\nThis is a preconfigured subclass of streaming transport. Writes to `sys.stderr` by default.\n\n**Class:** `mailers.transports.ConsoleTransport`\n**DSN:** `console://`\n**Options:**\n\n* `output` (typing.IO) - a writeable stream\n\n### Multi transport\n\nThe purpose of this transport is to provide a developer an option to provide a fallback transport.\nYou can configure several channels and `MultiTransport` will guarantee that at least one will deliver the message.\n\n**Class:** `mailers.transports.MultiTransport`\n**DSN:** `-`\n**Options:**\n\n* `transports` (list[Transport]) - subtransports\n\n### Custom transports.\n\nEach transport must extend `mailers.transports.Transport` base class.\n\n```python\nfrom email.message import Message\nfrom mailers import Mailer, Transport\n\n\nclass PrintTransport(Transport):\n    async def send(self, message: Message) -> None:\n        print(str(message))\n\n\nmailer = Mailer(PrintTransport())\n```\n',
    'author': 'alex.oleshkevich',
    'author_email': 'alex.oleshkevich@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/alex-oleshkevich/mailers',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
