import yaml
from jinja2 import Environment, FileSystemLoader
import pathlib


def render_template(template_name, servers_path, vars={}):
    """
    Function in charge of generating file content based on:
    - a template file name, that template file must reside into data/templates.
    - path to the servers.yml file, this file must has been generated by
      terraform.
    - additional variables passed through vars.

    Generated content is returned.
    """
    # Load servers.yml content
    servers_file = open(servers_path, 'r')
    servers = yaml.load(servers_file, Loader=yaml.FullLoader)

    # Template directory is located in ./data/templates
    current_dir = pathlib.Path(__file__).parent.resolve()
    templates_dir = pathlib.PurePath.joinpath(current_dir, 'data/templates')

    file_loader = FileSystemLoader(templates_dir)
    env = Environment(loader=file_loader, trim_blocks=True)
    template = env.get_template(template_name)

    return template.render(servers=servers['servers'], vars=vars)


def build_inventory_yml(dest, servers_path, vars={}):
    inventory = render_template('inventory.yml.j2', servers_path, vars)
    with open(dest, 'w') as f:
        f.write(inventory)


def build_config_yml(dest, servers_path, vars={}):
    config = render_template('config.yml.j2', servers_path, vars)
    with open(dest, 'w') as f:
        f.write(config)
