from typing import Any, Dict, Tuple, Type
from collections import defaultdict
from copy import deepcopy
import aerospike
import time


AerospikeKey: Type = Tuple[str, str, str]
OneMonth: int = 2629800  # one month in seconds since that's the default for our installation


class AerospikeClient :

	def __init__(self: 'AerospikeClient', config: Dict[str, Any] = None) :
		# tbh we really don't give a shit about the config since we're not setting anything up
		self._data = { }
		self._ttl = { }
		self.calls = defaultdict(lambda : [])


	def clear(self: 'AerospikeClient') :
		self._data.clear()
		self._ttl.clear()
		self.calls.clear()


	def __assert_key_type__(self: 'AerospikeClient', key: AerospikeKey) :
		assert type(key) == tuple
		assert len(key) == 3
		assert tuple(map(type, key)) == (str, str, str)


	def __assert_data_type__(self: 'AerospikeClient', data: Dict[str, Any]) :
		assert set(map(type, data.keys())) == { str }


	def put(self: 'AerospikeClient', key: AerospikeKey, data: Dict[str, Any], meta: Dict[str, Any] = None, policy: Dict[str, Any] = None) :
		self.calls['put'].append((key, data, meta, policy))
		self.__assert_key_type__(key)
		self.__assert_data_type__(data)

		ttl: int = int(meta['ttl'] if meta and 'ttl' in meta else OneMonth) or OneMonth

		# a couple of these I'm not sure what they're supposed to be
		# gen is set via meta or policy and the None I'm not sure of, but don't really care atm
		self._data[key] = ((key[0], key[1], None, hash(key).to_bytes(8, 'big', signed=True)), { 'ttl': ttl, 'gen': 1 }, data)
		self._ttl[key] = time.time()


	def get(self: 'AerospikeClient', key: AerospikeKey) :
		self.calls['get'].append((key))
		self.__assert_key_type__(key)
		ex = aerospike.exception.RecordNotFound(2, '127.0.0.1:3000 AEROSPIKE_ERR_RECORD_NOT_FOUND', 'src/main/client/get.c', 118, False)

		if key not in self._data :
			raise ex

		data = deepcopy(self._data[key])
		data[1]['ttl'] = round(max(data[1]['ttl'] - (time.time() - self._ttl[key]), 0))

		if data[1]['ttl'] <= 0 and data[1]['ttl'] != -1 :
			raise ex

		return data


	def increment(self: 'AerospikeClient', key: AerospikeKey, bin: str, value: int, meta: Dict[str, Any] = None, policy: Dict[str, Any] = None) :
		self.calls['increment'].append((key, bin, value, meta, policy))
		self.__assert_key_type__(key)
		assert type(bin) == str

		ttl: int = int(meta['ttl'] if meta and 'ttl' in meta else OneMonth) or OneMonth

		if key not in self._data :
			# a couple of these I'm not sure what they're supposed to be
			# gen is set via meta or policy and the None I'm not sure of, but don't really care atm
			self._data[key] = ((key[0], key[1], None, hash(key).to_bytes(8, 'big')), { 'ttl': ttl, 'gen': 1 }, { bin: value })
			self._ttl[key] = time.time()
			return

		if bin not in self._data[key][2] :
			self._data[key][2][bin] = value
			self._data[key][1]['ttl'] = ttl
			self._ttl[key] = time.time()
			return

		self._data[key][2][bin] += value
		self._data[key][1]['ttl'] = ttl
		self._ttl[key] = time.time()
