"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Flux = void 0;
const constructs_1 = require("constructs");
const k8s_1 = require("../imports/k8s");
class Flux extends constructs_1.Construct {
    constructor(scope, constructId, options) {
        super(scope, constructId);
        const label = { name: options.name };
        const limits = {
            memory: k8s_1.Quantity.fromString('256Mi'),
            cpu: k8s_1.Quantity.fromString('150m'),
        };
        const fluxPort = 3030;
        new k8s_1.Namespace(this, options.ns, {
            metadata: {
                name: options.ns,
                labels: label,
            },
        });
        const serviceAccount = new k8s_1.ServiceAccount(this, options.name + 'sa', {
            metadata: {
                labels: label,
                name: options.name,
                namespace: options.ns,
            },
        });
        const clusterRole = new k8s_1.ClusterRole(this, options.name + '-cr', {
            metadata: {
                labels: label,
                name: options.name,
            },
            rules: [
                {
                    apiGroups: ['*'],
                    resources: ['*'],
                    verbs: ['*'],
                },
                {
                    nonResourceURLs: ['*'],
                    verbs: ['*'],
                },
            ],
        });
        new k8s_1.ClusterRoleBinding(this, options.name + '-crb', {
            metadata: {
                labels: label,
                name: options.name,
            },
            roleRef: {
                apiGroup: 'rbac.authorization.k8s.io',
                kind: clusterRole.kind,
                name: clusterRole.name,
            },
            subjects: [
                {
                    kind: serviceAccount.kind,
                    name: serviceAccount.name,
                    namespace: options.ns,
                },
            ],
        });
        const probe = {
            httpGet: {
                port: fluxPort,
                path: 'api/flux/v6/identity.pub',
            },
            initialDelaySeconds: 5,
            timeoutSeconds: 5,
        };
        const fluxGitDeploy = options.name + '-git-deploy';
        const volumes = [
            {
                name: 'git-key',
                secret: {
                    secretName: fluxGitDeploy,
                    defaultMode: 0o400,
                },
            },
            {
                name: 'git-keygen',
                emptyDir: { medium: 'Memory' },
            },
        ];
        new k8s_1.Secret(this, options.name + '-secret', {
            metadata: {
                name: fluxGitDeploy,
                namespace: options.ns,
            },
            type: 'Opaque',
        });
        new k8s_1.Deployment(this, options.name + '-dp', {
            metadata: {
                namespace: options.ns,
                name: options.name,
            },
            spec: {
                replicas: 1,
                selector: {
                    matchLabels: label,
                },
                strategy: {
                    type: 'Recreate',
                },
                template: {
                    metadata: {
                        labels: label,
                    },
                    spec: {
                        serviceAccountName: serviceAccount.name,
                        volumes: volumes,
                        containers: [
                            {
                                name: options.name,
                                image: options.image + ':' + options.tag,
                                imagePullPolicy: 'IfNotPresent',
                                ports: [{ containerPort: fluxPort }],
                                resources: {
                                    limits: limits,
                                },
                                livenessProbe: probe,
                                readinessProbe: probe,
                                volumeMounts: [
                                    {
                                        name: volumes[0].name,
                                        mountPath: '/etc/fluxd/ssh',
                                        readOnly: true,
                                    },
                                    {
                                        name: volumes[1].name,
                                        mountPath: '/etc/fluxd/keygen',
                                    },
                                ],
                                args: options.arguments,
                            },
                        ],
                    },
                },
            },
        });
        const memcachedName = 'memcached';
        const memcachedLabels = {
            name: memcachedName,
        };
        const memcachedPort = 11211;
        new k8s_1.Deployment(this, options.name + '-memcached', {
            metadata: {
                name: memcachedName,
                namespace: options.ns,
            },
            spec: {
                replicas: 1,
                selector: {
                    matchLabels: memcachedLabels,
                },
                template: {
                    metadata: {
                        labels: memcachedLabels,
                    },
                    spec: {
                        containers: [
                            {
                                name: memcachedName,
                                image: 'memcached:' + options.memcachedTag,
                                args: ['-m 512', '-I 5m', '-p ' + memcachedPort],
                                ports: [
                                    {
                                        name: 'clients',
                                        containerPort: memcachedPort,
                                    },
                                ],
                                securityContext: {
                                    runAsGroup: memcachedPort,
                                    runAsUser: memcachedPort,
                                    allowPrivilegeEscalation: false,
                                },
                            },
                        ],
                    },
                },
            },
        });
        new k8s_1.Service(this, memcachedName + '-svc', {
            metadata: {
                name: memcachedName,
                namespace: options.name,
            },
            spec: {
                ports: [
                    {
                        port: memcachedPort,
                        name: memcachedName,
                    },
                ],
                selector: {
                    ['name']: memcachedName,
                },
            },
        });
    }
}
exports.Flux = Flux;
//# sourceMappingURL=data:application/json;base64,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