"""Width and measurement utilities.

- visible_width: grapheme-aware width with ANSI/markup stripping and tabs
- measure_text: (width, height) for multi-line text

Escapes handled:
- ANSI SGR sequences (e.g., "\x1b[31m", reset "\x1b[0m") are ignored for width
- Inline markup in square-bracket form (e.g., "[red]text[reset]") is ignored when requested
- Soft hyphen (U+00AD) has zero width; display handling is left to wrappers
- Tab ("\t") advances to the next tab stop (default 4 columns)
"""

from __future__ import annotations

import re
from typing import TYPE_CHECKING

from .graphemes import grapheme_width, iter_graphemes

if TYPE_CHECKING:
    TAB_SENTINEL: int

__all__ = [
    "strip_ansi",
    "strip_markup",
    "cell_width",
    "visible_width",
    "measure_text",
]

_ANSI_RE = re.compile(r"\033\[[0-9;]*m")
_MARKUP_SQUARE_RE = re.compile(r"\[[^\]]+\]")
_MARKUP_CURLY_RE = re.compile(r"\{[^}]+\}")


def strip_ansi(text: str) -> str:
    """Strip ANSI escape sequences from text."""
    return _ANSI_RE.sub("", text)


def strip_markup(text: str) -> str:
    """Strip inline markup tags (square- or curly-braced) from text."""
    text = _MARKUP_SQUARE_RE.sub("", text)
    text = _MARKUP_CURLY_RE.sub("", text)
    return text


def _visible_width_no_controls(text: str, *, tab_size: int) -> int:
    """Compute width without ANSI/markup, honoring tabs and soft hyphens.

    This function assumes `text` has had markup and ANSI removed already.
    """
    width = 0
    for cluster in iter_graphemes(text):
        if cluster == "\t":
            if tab_size <= 0:
                continue
            # advance to next tab stop
            next_stop = ((width // tab_size) + 1) * tab_size
            width = next_stop
            continue
        if cluster == "\u00AD":  # soft hyphen
            # zero width in measurement
            continue
        width += grapheme_width(cluster)
    return width


def cell_width(text: str, *, tab_size: int = 4) -> int:
    """Return total terminal width of `text`.

    Equivalent to visible_width(text, ignore_ansi=True, ignore_markup=True).
    """
    return visible_width(text, ignore_ansi=True, ignore_markup=True, tab_size=tab_size)


def visible_width(
    text: str,
    *,
    ignore_ansi: bool = True,
    ignore_markup: bool = True,
    tab_size: int = 4,
) -> int:
    """Return display width of `text`.

    Args:
        text: Input text
        ignore_ansi: If True, strip ANSI SGR sequences before measuring
        ignore_markup: If True, strip inline markup like "[red]..." and "{tag:payload}"
        tab_size: Size of tab stops used for "\t" expansion (0 disables)
    """
    working = text
    if ignore_markup:
        working = strip_markup(working)
    if ignore_ansi:
        working = strip_ansi(working)
    return _visible_width_no_controls(working, tab_size=tab_size)


def measure_text(text: str, *, tab_size: int = 4) -> tuple[int, int]:
    """Return (width, height) for the supplied text.

    Width is the maximum visible width across all lines.
    Height is the number of lines, accounting for splitlines() semantics.
    """
    # ANSI is stripped, markup left intact or stripped? We strip both to
    # ensure width corresponds to on-screen cells.
    raw = strip_ansi(text)
    lines = raw.splitlines() or [""]
    width = 0
    for line in lines:
        # When measuring, ignore markup so width reflects final cells.
        width = max(width, visible_width(line, ignore_ansi=False, ignore_markup=True, tab_size=tab_size))
    return width, len(lines)
