"""
Text manipulation utilities for Textforge.
"""

from __future__ import annotations

import re
from dataclasses import dataclass
from typing import Literal

from .. import text_engine
from ..style.colors import Color
from ..symbols import Symbols

_ANSI_RE = re.compile(r"\033\[[0-9;]*m")
_BREAKING_SPACES = {" ", "\t", "\u3000"}


@dataclass(slots=True)
class _Token:
    raw: str
    type: Literal["grapheme", "control"]
    width: int = 0

    @property
    def is_break(self) -> bool:
        return self.type == "grapheme" and self.raw in _BREAKING_SPACES


def _tokenize(text: str) -> list[_Token]:
    tokens: list[_Token] = []
    i = 0
    length = len(text)
    while i < length:
        ch = text[i]
        if ch == "{":
            end = text.find("}", i + 1)
            if end != -1:
                tokens.append(_Token(text[i : end + 1], "control"))
                i = end + 1
                continue
        match = _ANSI_RE.match(text, i)
        if match:
            tokens.append(_Token(match.group(0), "control"))
            i = match.end()
            continue
        grapheme, next_index = text_engine.next_grapheme(text, i)
        if not grapheme:
            break
        tokens.append(_Token(grapheme, "grapheme", text_engine.grapheme_width(grapheme)))
        i = next_index
    return tokens


def wrap_text(text: str, width: int) -> list[str]:
    """Wrap text to specified width while preserving markup and ANSI codes."""
    if width <= 0:
        return [text]

    lines: list[str] = []

    for paragraph in text.split("\n"):
        tokens = _tokenize(paragraph)
        if not tokens:
            lines.append("")
            continue

        current: list[_Token] = []
        current_width = 0
        last_break: int | None = None
        idx = 0

        while idx < len(tokens):
            token = tokens[idx]
            token_width = token.width if token.type == "grapheme" else 0
            exceeds = (
                current_width > 0
                and token_width > 0
                and current_width + token_width > width
            )

            if exceeds:
                break_idx = last_break if last_break is not None else len(current)
                line_tokens = current[:break_idx]
                remainder = current[break_idx:]
                while remainder and remainder[0].type == "grapheme" and remainder[0].raw.strip() == "":
                    remainder.pop(0)
                lines.append("".join(tok.raw for tok in line_tokens).rstrip())
                current = remainder
                current_width = sum(tok.width for tok in current if tok.type == "grapheme")
                last_break = None
                continue  # Re-evaluate current token

            current.append(token)
            if token_width:
                current_width += token_width
                if token.is_break:
                    last_break = len(current)
            idx += 1

        if current:
            lines.append("".join(tok.raw for tok in current).rstrip())

    return lines


def strip_ansi(text: str) -> str:
    """Remove ANSI color codes from text for length calculation."""
    return text_engine.strip_ansi(text)


def get_visible_length(text: str) -> int:
    """Return grapheme-aware display width excluding markup and ANSI codes."""
    return text_engine.visible_width(text, ignore_ansi=True, ignore_markup=True)


def center_text(text: str, width: int, fill_char: str = " ") -> str:
    """
    Center each line of `text` within the given width using `fill_char`.
    """
    result_lines: list[str] = []
    for line in text.split("\n"):
        line_width = get_visible_length(line)
        if line_width >= width:
            result_lines.append(line)
            continue
        pad_total = width - line_width
        left = pad_total // 2
        right = pad_total - left
        result_lines.append(f"{fill_char * left}{line}{fill_char * right}")
    return "\n".join(result_lines)


def format_text(
    display_text: str,
    width: int | None = None,
    blank_lines: tuple[int, int] | None = None,
    color: str | None = None
) -> None:
    """
    Format text with optional width, blank lines, and color.
    Supports inline markup: [color_name]text[reset]

    Args:
        display_text: The text to format (supports inline markup like [red]text[reset])
        width: The width of the display
        blank_lines: A tuple of the number of blank lines before and after the display text
        color: Default color for the entire text (can be overridden by inline markup)
    """
    formatted_text = Color.apply_inline_markup(display_text)

    if color and "{" not in display_text:
        text_color = Color.get_color(color)
        formatted_text = text_color + formatted_text + Color.RESET

    if width:
        visible_len = get_visible_length(formatted_text)
        total_pad = max(0, width - visible_len)
        left_pad = total_pad // 2
        right_pad = total_pad - left_pad
        formatted_text = (" " * left_pad) + formatted_text + (" " * right_pad)

    if blank_lines:
        for _ in range(blank_lines[0]):
            print()
        print(formatted_text + Color.RESET)
        for _ in range(blank_lines[1]):
            print()
    else:
        print(formatted_text + Color.RESET)


def wrap_text_advanced(
    text: str,
    width: int,
    indent_first: int = 0,
    indent_rest: int = 0,
    preserve_paragraphs: bool = True
) -> list[str]:
    """
    Advanced text wrapping with indentation options.

    Args:
        text: Text to wrap
        width: Maximum line width
        indent_first: Indentation for first line
        indent_rest: Indentation for subsequent lines
        preserve_paragraphs: Preserve paragraph breaks

    Returns:
        List of wrapped lines
    """
    if preserve_paragraphs:
        paragraphs = text.split("\n\n")
        result: list[str] = []
        for para in paragraphs:
            wrapped = wrap_text(para, width - indent_first)
            if wrapped:
                result.append(" " * indent_first + wrapped[0])
                for line in wrapped[1:]:
                    result.append(" " * indent_rest + line)
                result.append("")
        return result[:-1] if result else []
    else:
        wrapped = wrap_text(text, width - indent_first)
        if not wrapped:
            return []
        result = [" " * indent_first + wrapped[0]]
        for line in wrapped[1:]:
            result.append(" " * indent_rest + line)
        return result


def create_gradient_text(
    text: str,
    colors: list[tuple[int, int, int]]
) -> str:
    """
    Create multi-color gradient text.

    Args:
        text: Text to apply gradient to
        colors: List of RGB color tuples for gradient stops

    Returns:
        Text with gradient applied
    """
    if len(colors) < 2:
        return text

    length = len(text)
    result = ""

    for i, char in enumerate(text):
        t = i / (length - 1) if length > 1 else 0
        segment_size = 1.0 / (len(colors) - 1)
        segment_idx = min(int(t / segment_size), len(colors) - 2)

        segment_t = (t - segment_idx * segment_size) / segment_size

        start_color = colors[segment_idx]
        end_color = colors[segment_idx + 1]

        r = int(start_color[0] + (end_color[0] - start_color[0]) * segment_t)
        g = int(start_color[1] + (end_color[1] - start_color[1]) * segment_t)
        b = int(start_color[2] + (end_color[2] - start_color[2]) * segment_t)

        result += f"\033[38;2;{r};{g};{b}m{char}"

    return result + "\033[0m"


def create_multi_gradient_text(text: str, stops: list[tuple[float, tuple[int, int, int]]]) -> str:
    """
    Create gradient text from multiple color stops.

    Args:
        text: Text to apply gradient to
        stops: List of (position, RGB) where position is in [0,1]
    """
    if not text or not stops:
        return text
    stops = sorted(stops, key=lambda s: s[0])
    result = ""
    for i, ch in enumerate(text):
        t = i / max(1, len(text) - 1)
        # find segment
        left_idx = 0
        right_idx = len(stops) - 1
        for si in range(len(stops) - 1):
            if stops[si][0] <= t <= stops[si + 1][0]:
                left_idx = si
                right_idx = si + 1
                break
        t0, c0 = stops[left_idx]
        t1, c1 = stops[right_idx]
        denom = max(1e-6, t1 - t0)
        local_t = (t - t0) / denom
        r = int(c0[0] + (c1[0] - c0[0]) * local_t)
        g = int(c0[1] + (c1[1] - c0[1]) * local_t)
        b = int(c0[2] + (c1[2] - c0[2]) * local_t)
        result += f"\033[38;2;{r};{g};{b}m{ch}"
    return result + "\033[0m"


def background_gradient_text(text: str, colors: list[tuple[int, int, int]]) -> str:
    """
    Apply a horizontal background gradient behind the text.
    """
    if len(colors) < 2:
        return text
    length = len(text)
    result = ""
    for i, char in enumerate(text):
        t = i / (length - 1) if length > 1 else 0
        segment_size = 1.0 / (len(colors) - 1)
        segment_idx = min(int(t / segment_size), len(colors) - 2)
        segment_t = (t - segment_idx * segment_size) / segment_size
        start_color = colors[segment_idx]
        end_color = colors[segment_idx + 1]
        r = int(start_color[0] + (end_color[0] - start_color[0]) * segment_t)
        g = int(start_color[1] + (end_color[1] - start_color[1]) * segment_t)
        b = int(start_color[2] + (end_color[2] - start_color[2]) * segment_t)
        result += f"\033[48;2;{r};{g};{b}m{char}\033[0m"
    return result


def radial_gradient_text(text: str, colors: list[tuple[int, int, int]]) -> str:
    """
    Apply a radial-like gradient across 1D text by blending from center outward.
    """
    if len(colors) < 2:
        return text
    n = len(text)
    if n == 0:
        return text
    mid = (n - 1) / 2.0
    # Build multi-stop segments across radius 0..1
    stops: list[tuple[float, tuple[int, int, int]]] = []
    for i, col in enumerate(colors):
        stops.append((i / (len(colors) - 1), col))

    def _color_at(t: float) -> tuple[int, int, int]:
        # find two surrounding stops
        left = 0
        right = len(stops) - 1
        for si in range(len(stops) - 1):
            if stops[si][0] <= t <= stops[si + 1][0]:
                left = si
                right = si + 1
                break
        t0, c0 = stops[left]
        t1, c1 = stops[right]
        local = 0.0 if t1 == t0 else (t - t0) / (t1 - t0)
        r = int(c0[0] + (c1[0] - c0[0]) * local)
        g = int(c0[1] + (c1[1] - c0[1]) * local)
        b = int(c0[2] + (c1[2] - c0[2]) * local)
        return (r, g, b)

    out = []
    for i, ch in enumerate(text):
        # radius from center normalized to 0..1
        r = abs(i - mid) / max(1.0, mid)
        color = _color_at(1.0 - r)
        out.append(f"\033[38;2;{color[0]};{color[1]};{color[2]}m{ch}")
    return "".join(out) + "\033[0m"


def angular_gradient_text(text: str, colors: list[tuple[int, int, int]], cycles: float = 1.0) -> str:
    """
    Apply an angular-like gradient along text by mapping index to a repeating hue cycle.
    """
    if len(colors) < 2:
        return text
    n = len(text)
    if n == 0:
        return text
    stops: list[tuple[float, tuple[int, int, int]]] = []
    for i, col in enumerate(colors):
        stops.append((i / (len(colors) - 1), col))

    def _color_at(t: float) -> tuple[int, int, int]:
        t = t % 1.0
        left = 0
        right = len(stops) - 1
        for si in range(len(stops) - 1):
            if stops[si][0] <= t <= stops[si + 1][0]:
                left = si
                right = si + 1
                break
        t0, c0 = stops[left]
        t1, c1 = stops[right]
        local = 0.0 if t1 == t0 else (t - t0) / (t1 - t0)
        r = int(c0[0] + (c1[0] - c0[0]) * local)
        g = int(c0[1] + (c1[1] - c0[1]) * local)
        b = int(c0[2] + (c1[2] - c0[2]) * local)
        return (r, g, b)

    out = []
    for i, ch in enumerate(text):
        t = (i / max(1, n - 1)) * cycles
        color = _color_at(t)
        out.append(f"\033[38;2;{color[0]};{color[1]};{color[2]}m{ch}")
    return "".join(out) + "\033[0m"


def box_text(
    text: str,
    width: int,
    padding: int = 1,
    border_style: str = "box",
    color: str | None = None
) -> str:
    """
    Box text and return as string (not printed).
    """
    symbols = Symbols.get_symbols(border_style)
    lines = wrap_text(text, width - 4 - padding * 2)

    result: list[str] = []
    result.append(symbols["top_left"] + symbols["horizontal"] * (width - 2) + symbols["top_right"])

    for _ in range(padding):
        result.append(symbols["vertical"] + " " * (width - 2) + symbols["vertical"])

    for line in lines:
        colored_line = Color.apply_inline_markup(line) if "{" in line else line
        if color:
            text_color = Color.get_color(color)
            colored_line = text_color + colored_line + Color.RESET

        visible_len = get_visible_length(colored_line)
        padding_needed = width - 4 - padding - visible_len
        content_line = (
            symbols["vertical"] + " " * padding + colored_line + " " * padding_needed + " " + symbols["vertical"]
        )
        result.append(content_line)

    for _ in range(padding):
        result.append(symbols["vertical"] + " " * (width - 2) + symbols["vertical"])

    result.append(symbols["bottom_left"] + symbols["horizontal"] * (width - 2) + symbols["bottom_right"])

    return "\n".join(result)


def justify_text(text: str, width: int) -> str:
    """
    Justify text to exact width by adjusting spaces.
    """
    words = text.split()
    if len(words) <= 1:
        return text

    total_word_length = sum(len(word) for word in words)
    total_spaces = width - total_word_length
    gaps = len(words) - 1

    if gaps == 0:
        return text

    spaces_per_gap = total_spaces // gaps
    extra_spaces = total_spaces % gaps

    result: list[str] = []
    for i, word in enumerate(words[:-1]):
        result.append(word)
        result.append(" " * (spaces_per_gap + (1 if i < extra_spaces else 0)))
    result.append(words[-1])

    return "".join(result)


def rainbow_text(text: str) -> str:
    """
    Apply rainbow colors to text.
    """
    colors = [
        (255, 0, 0),
        (255, 127, 0),
        (255, 255, 0),
        (0, 255, 0),
        (0, 0, 255),
        (75, 0, 130),
        (148, 0, 211),
    ]

    return create_gradient_text(text, colors)


def indent_text(text: str, indent: int, first_line_indent: int | None = None) -> str:
    """
    Indent text with optional different first line indent.
    """
    if first_line_indent is None:
        first_line_indent = indent

    lines = text.split("\n")
    if not lines:
        return text

    result = [" " * first_line_indent + lines[0]]
    for line in lines[1:]:
        result.append(" " * indent + line)

    return "\n".join(result)


def truncate_text(text: str, max_length: int, suffix: str = "...") -> str:
    """
    Truncate text to maximum length with suffix.
    """
    if max_length <= 0:
        return ""

    visible = get_visible_length(text)
    if visible <= max_length:
        return text

    suffix_width = get_visible_length(suffix)
    if suffix_width >= max_length:
        return suffix

    tokens = _tokenize(text)
    acc: list[str] = []
    width = 0
    limit = max_length - suffix_width

    for token in tokens:
        if token.type == "grapheme":
            if width + token.width > limit:
                break
            width += token.width
        acc.append(token.raw)

    return "".join(acc) + suffix
