"""Profiling helpers for Textforge.

Provides a context manager and decorator to measure elapsed time of code blocks
and callables. Designed to be lightweight and dependency-free.
"""

from __future__ import annotations

import functools
import time
from contextlib import contextmanager
from typing import TYPE_CHECKING, Any, TypeVar, cast

if TYPE_CHECKING:
    from collections.abc import Callable, Generator

T = TypeVar("T")


@contextmanager
def time_block(label: str) -> Generator[None, Any, Any]:
    """Measure a code block and print duration with the given label."""
    start = time.perf_counter()
    try:
        yield
    finally:
        dur_ms = (time.perf_counter() - start) * 1000
        print(f"{label}: {dur_ms:.2f} ms")


def time_callable(label: str | None = None) -> Callable[[Callable[..., T]], Callable[..., T]]:
    """Decorator that measures function execution time and prints it.

    Args:
        label: Optional label. If omitted, the function's __name__ is used.
    """

    def _decorator(func: Callable[..., T]) -> Callable[..., T]:
        name = label or func.__name__

        @functools.wraps(func)
        def _wrapped(*args: Any, **kwargs: Any) -> T:
            start = time.perf_counter()
            try:
                return cast("T", func(*args, **kwargs))
            finally:
                dur_ms = (time.perf_counter() - start) * 1000
                print(f"{name}: {dur_ms:.2f} ms")

        return _wrapped

    return _decorator
