"""Logging integration for Textforge.

Provides `TextforgeHandler`, a logging.Handler subclass that formats messages
with timestamps, levels, and optional colorization using Textforge's Color.
"""

from __future__ import annotations

import logging
from typing import TextIO

from ..style.colors import Color


class TextforgeFormatter(logging.Formatter):
    """Simple formatter that applies color by level and markup rendering."""

    LEVEL_COLORS = {
        logging.DEBUG: "muted",
        logging.INFO: "info",
        logging.WARNING: "warning",
        logging.ERROR: "error",
        logging.CRITICAL: "error",
    }

    def format(self, record: logging.LogRecord) -> str:
        level_color = self.LEVEL_COLORS.get(record.levelno, "fg")
        level = logging.getLevelName(record.levelno)
        color_code = Color.get_color(level_color)
        reset = Color.RESET
        # Render basic template: [time][level] message
        # Time formatted via logging's asctime when datefmt provided
        time = self.formatTime(record, self.datefmt)
        msg = super().format(record)
        # Ensure the line ends with a reset to avoid color bleed
        return f"{Color.DIM}[{time}] {reset}{color_code}{level:>8}{reset} {msg}{reset}"


class TextforgeHandler(logging.StreamHandler):
    """Logging handler that outputs ANSI-colored logs compatible with Textforge."""

    def __init__(self, stream: TextIO | None = None) -> None:
        super().__init__(stream)
        fmt = TextforgeFormatter("%(message)s", datefmt="%H:%M:%S")
        self.setFormatter(fmt)


def get_logger(name: str = "textforge", level: int = logging.INFO) -> logging.Logger:
    """Create or fetch a logger preconfigured with TextforgeHandler."""
    logger = logging.getLogger(name)
    if not any(isinstance(h, TextforgeHandler) for h in logger.handlers):
        logger.addHandler(TextforgeHandler())
    logger.setLevel(level)
    logger.propagate = False
    return logger
