"""Lightweight lint rules for Textforge content and markup."""

from __future__ import annotations

import re

from .accessibility import is_contrast_sufficient

_TAG_RE = re.compile(r"\[([^\]]+)\]")


def lint_markup(text: str) -> list[str]:
    """Check for unclosed reset tags and unknown tags pattern-wise."""
    warnings: list[str] = []
    opens = 0
    for m in _TAG_RE.finditer(text):
        token = m.group(1).strip()
        if token.lower() == "reset":
            if opens == 0:
                warnings.append("reset without prior tag")
            else:
                opens -= 1
        else:
            opens += 1
    if opens > 0:
        warnings.append("missing [reset] tag(s)")
    return warnings


def lint_contrast(fg_hex: str, bg_hex: str, large_text: bool = False) -> list[str]:
    if is_contrast_sufficient(fg_hex, bg_hex, large_text=large_text):
        return []
    return [f"insufficient contrast {fg_hex} on {bg_hex}"]
