"""I/O toolkit helpers: reading/writing files and stdin."""

from __future__ import annotations

import sys
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from collections.abc import Iterable


def read_text(path: str, encoding: str = "utf-8") -> str:
    with open(path, encoding=encoding) as f:
        return f.read()


def write_text(path: str, text: str, encoding: str = "utf-8") -> None:
    with open(path, "w", encoding=encoding) as f:
        f.write(text)


def read_stdin(encoding: str | None = None) -> str:
    data = sys.stdin.buffer.read()
    enc = encoding or getattr(sys.stdin, "encoding", None) or "utf-8"
    return data.decode(enc, errors="replace")


def iter_stdin_lines() -> Iterable[str]:
    for line in sys.stdin:
        yield line.rstrip("\n")


# Module-level storage for stdin restoration
_original_stdin = None


def ensure_windows_valid_stdin() -> None:
    """On Windows, ensure sys.stdin is attached to a valid OS handle.

    Some CI/test harnesses replace stdin with an object that lacks a real
    handle, which can cause subprocess spawning with capture_output to fail
    (WinError 6) when the OS tries to duplicate inheritable handles.
    If stdin lacks a valid handle, reattach it to the console input device.
    """
    global _original_stdin
    try:
        if sys.platform.startswith("win"):
            # If stdin is an in-memory stream (e.g., StringIO used by tests),
            # do not mutate it; it's already a valid stream for our purposes.
            try:
                import io as _io
                if isinstance(sys.stdin, _io.StringIO) or not hasattr(sys.stdin, "fileno"):
                    return
            except Exception:
                return
            try:
                # Verify current stdin has a valid OS handle
                import msvcrt

                _ = msvcrt.get_osfhandle(sys.stdin.fileno())
                return
            except Exception:
                # Reattach to a valid handle: prefer console input, fallback to NUL
                if _original_stdin is None:
                    _original_stdin = sys.stdin
                conin = None
                try:
                    conin = open("CONIN$", encoding=getattr(sys.stdin, "encoding", "utf-8"), errors="ignore")
                except OSError:
                    try:
                        conin = open("NUL")
                    except OSError:
                        conin = None
                if conin is not None:
                    sys.stdin = conin
                else:
                    return
    except Exception:
        # Best effort only
        pass


def restore_stdin() -> None:
    """Restore the original stdin if it was reassigned by ensure_windows_valid_stdin."""
    global _original_stdin
    if _original_stdin is not None:
        sys.stdin = _original_stdin
        _original_stdin = None
