"""Keyboard input helpers for interactive console components."""
from __future__ import annotations

import sys
import time
from typing import Final

__all__ = ["read_key"]

_WIN_PLATFORM: Final[bool] = sys.platform.startswith("win")


def read_key(timeout: float | None = None, *, allow_fallback: bool = True) -> str:
    """Read a single key press and return a semantic token.

    Returns identifiers such as ``"up"``, ``"down"``, ``"enter"``, ``"escape"``,
    ``"page_up"``, ``"page_down"``, or individual printable characters
    lowercased. When a timeout (seconds) is provided the function returns an
    empty string on expiry.
    """

    if _WIN_PLATFORM:
        try:
            import msvcrt
        except ImportError:
            pass
        else:
            deadline = (time.perf_counter() + timeout) if timeout is not None else None
            mapping = {
                "H": "up",
                "P": "down",
                "K": "left",
                "M": "right",
                "I": "page_up",
                "Q": "page_down",
                "G": "home",
                "O": "end",
            }
            while True:
                if deadline is not None and not msvcrt.kbhit():
                    if time.perf_counter() >= deadline:
                        return ""
                    time.sleep(0.01)
                    continue
                ch = msvcrt.getwch()
                if ch in ("\r", "\n"):
                    return "enter"
                if ch == "\x1b":
                    return "escape"
                if ch in ("\x00", "\xe0"):
                    code = msvcrt.getwch()
                    return mapping.get(code, "")
                if ch == "\x08":
                    return "backspace"
                if ch == "\t":
                    return "tab"
                if ch == " ":
                    return "space"
                if ch in ("\x03", "\x04"):
                    raise KeyboardInterrupt
                return ch.lower()

    # GUI backend queue path
    try:
        from ..core.backends import get_gui_key_queue  # lazy import to avoid backend cost
    except Exception:
        get_gui_key_queue = None
    if get_gui_key_queue is not None:
        q = get_gui_key_queue()
        if q is not None:
            try:
                if timeout is None:
                    token = q.get(block=True)
                else:
                    token = q.get(timeout=max(0.0, timeout))
                return token
            except Exception:
                return ""

    # POSIX or fallback path
    try:
        import select
        import termios
        import tty
    except Exception:
        if not allow_fallback:
            raise
    else:
        try:
            fd = sys.stdin.fileno()
        except (AttributeError, ValueError, OSError):
            fd = -1

        if fd >= 0:
            if timeout is not None:
                ready, _, _ = select.select([sys.stdin], [], [], timeout)
                if not ready:
                    return ""
            old_settings = termios.tcgetattr(fd)
            try:
                tty.setraw(fd)
                first = sys.stdin.read(1)
                if first == "\x1b":  # ESC sequence or bare escape
                    # Read as much as we can within a small window to capture sequences
                    seq = ""
                    # small wait to accumulate the rest of the sequence
                    while select.select([sys.stdin], [], [], 0.01)[0]:
                        ch = sys.stdin.read(1)
                        seq += ch
                        if ch.isalpha() or ch == "~":
                            break
                    # Common CSI (ESC [) and SS3 (ESC O) mappings
                    keymap = {
                        "[A": "up",
                        "[B": "down",
                        "[C": "right",
                        "[D": "left",
                        "[H": "home",
                        "[F": "end",
                        "[1~": "home",
                        "[4~": "end",
                        "[2~": "insert",
                        "[3~": "delete",
                        "[5~": "page_up",
                        "[6~": "page_down",
                        "OP": "f1",
                        "OQ": "f2",
                        "OR": "f3",
                        "OS": "f4",
                    }
                    # Normalize sequences like ESC [ 5 ~ => "[5~"
                    seq_norm = seq.replace(" ", "")
                    if seq_norm.startswith("[") or seq_norm.startswith("O"):
                        token = seq_norm
                    else:
                        token = seq_norm
                    return keymap.get(token, "escape")
                if first in ("\r", "\n"):
                    return "enter"
                if first == "\t":
                    return "tab"
                if first == " ":
                    return "space"
                if first == "\x7f":
                    return "backspace"
                return first.lower()
            finally:
                termios.tcsetattr(fd, termios.TCSADRAIN, old_settings)

    if not allow_fallback:
        return ""

    try:
        data = input()
    except EOFError:
        return ""
    data = data.strip()
    if not data:
        return "enter"
    return data.lower()
