"""Minimal i18n helper with a process-wide translator and a [t:key] tag.

This module provides a very small translation registry that maps locales to
key->string dictionaries, plus helpers to switch the active locale and fetch
translations.
"""

from __future__ import annotations

from dataclasses import dataclass, field


@dataclass(slots=True)
class Translator:
    locale: str = "en"
    catalogs: dict[str, dict[str, str]] = field(default_factory=dict)

    def set_locale(self, locale: str) -> None:
        self.locale = locale

    def register(self, locale: str, mapping: dict[str, str]) -> None:
        self.catalogs.setdefault(locale, {}).update(mapping)

    def translate(self, key: str) -> str:
        return self.catalogs.get(self.locale, {}).get(key, key)


_translator = Translator()


def get_translator() -> Translator:
    return _translator


def set_locale(locale: str) -> None:
    _translator.set_locale(locale)


def register_translations(locale: str, mapping: dict[str, str]) -> None:
    _translator.register(locale, mapping)


def t(key: str) -> str:
    return _translator.translate(key)
