"""Deprecation utilities for Textforge public API.

This module provides a simple decorator to mark functions, methods,
classes, and properties as deprecated. It emits a `DeprecationWarning`
on first use per-callsite and preserves metadata for introspection.
"""

from __future__ import annotations

import functools
import inspect
import warnings
from collections.abc import Callable
from typing import Any, TypeVar

F = TypeVar("F", bound=Callable[..., Any])


def deprecated(reason: str, *, since: str, remove_in: str) -> Callable[[Any], Any]:
    """Mark a callable as deprecated.

    Args:
        reason: Short human-readable message or migration path.
        since: Version when the deprecation was introduced (e.g., "1.1").
        remove_in: Target version for removal (e.g., "2.0").

    Returns:
        A decorator that wraps the given callable and emits a DeprecationWarning
        on first use.
    """

    def decorator(obj: Any) -> Any:
        name = getattr(obj, "__qualname__", getattr(obj, "__name__", str(obj)))
        message = (
            f"{name} is deprecated since {since} and will be removed in {remove_in}. "
            f"{reason}".strip()
        )

        # Classes and callables need different wrapping strategies
        if inspect.isclass(obj):
            original_init = getattr(obj, "__init__", None)

            def __init__(self: Any, *args: Any, **kwargs: Any) -> None:
                warnings.warn(message, DeprecationWarning, stacklevel=2)
                if original_init is not None:
                    original_init(self, *args, **kwargs)

            obj.__init__ = __init__
            return obj

        @functools.wraps(obj)
        def wrapper(*args: Any, **kwargs: Any):
            warnings.warn(message, DeprecationWarning, stacklevel=2)
            return obj(*args, **kwargs)

        return wrapper

    return decorator


__all__ = ["deprecated"]
