"""Lightweight configuration loader for Textforge.

Recognizes a simple key=value format in `tf.config` or `textforge.config` at
the project root (current working directory or any parent). Blank lines and
lines starting with `#` are ignored. Later files override earlier ones if both
exist in the same directory hierarchy.

Known keys:
- renderer: one of "tty" (default), "cli", "console", "terminal", "gui"
"""

from __future__ import annotations

import os

__all__ = ["load_textforge_config", "get_config_value", "get_default_renderer"]


def _find_config_paths(start: str | None = None) -> list[str]:
    root = os.path.abspath(start or os.getcwd())
    paths: list[str] = []
    current = root
    while True:
        tf_path = os.path.join(current, "tf.config")
        tfg_path = os.path.join(current, "textforge.config")
        if os.path.isfile(tf_path):
            paths.append(tf_path)
        if os.path.isfile(tfg_path):
            paths.append(tfg_path)
        parent = os.path.dirname(current)
        if parent == current:
            break
        current = parent
    return paths


def _parse_config_file(path: str) -> dict[str, str]:
    values: dict[str, str] = {}
    try:
        with open(path, encoding="utf-8") as f:
            for raw in f:
                line = raw.strip()
                if not line or line.startswith("#"):
                    continue
                # support simple key=value; ignore malformed lines safely
                if "=" in line:
                    k, v = line.split("=", 1)
                    key = k.strip().lower()
                    val = v.strip()
                    values[key] = val
    except Exception:
        return {}
    return values


def load_textforge_config(start: str | None = None) -> dict[str, str]:
    values: dict[str, str] = {}
    for path in reversed(_find_config_paths(start)):
        # later in the list means closer to root; reversed to let nearer files win
        values.update(_parse_config_file(path))
    return values


def get_config_value(key: str, default: str | None = None) -> str | None:
    cfg = load_textforge_config()
    return cfg.get(key.lower(), default)


def get_default_renderer() -> str:
    value = (get_config_value("renderer") or "").strip().lower()
    if value in ("cli", "console", "tty"):
        return "tty"
    if value in ("terminal", "term", "custom_terminal"):
        return "terminal"
    if value in ("gui", "window", "desktop"):
        return "gui"
    return "tty"
