"""Benchmark suite and metrics for Textforge.

Provides:
- run_suite(): representative component rendering benchmarks
- run_micro_suite(): micro-benchmarks for markup, gradient, layout, vdom diff
- metrics helpers: import time, first render latency, steady-state live update
"""

from __future__ import annotations

import io
import subprocess
import time
from dataclasses import dataclass
from typing import TYPE_CHECKING, Any, TextIO

from ..components import Paragraph, ProgressBar, Table
from ..core import Console
from ..core.vdom import diff_changes
from ..layout.engine import LayoutNode, LayoutStyle, compute_layout
from ..markup import MarkupEngine
from ..style.colors import Color
from .profiling import time_callable

if TYPE_CHECKING:
    from collections.abc import Callable


@dataclass(slots=True)
class BenchResult:
    name: str
    ms: float
    iters: int


@time_callable("bench")
def _bench(name: str, func: Callable[[], None], iters: int = 50) -> BenchResult:
    console = Console()
    start = time.perf_counter()
    for _ in range(iters):
        console.print(func, end="")  # capture rendering via CallableRenderable
    ms = (time.perf_counter() - start) * 1000.0
    return BenchResult(name, ms, iters)


def _bench_call(name: str, func: Callable[[], Any], iters: int = 1000) -> BenchResult:  # Any: benchmark functions can return any type
    """Tight loop benchmark that invokes ``func`` without Console indirection."""
    start = time.perf_counter()
    for _ in range(iters):
        func()
    ms = (time.perf_counter() - start) * 1000.0
    return BenchResult(name, ms, iters)


def run_suite() -> list[BenchResult]:
    """Run a small, representative suite of rendering tasks."""
    results: list[BenchResult] = []

    def p_small() -> None:
        Paragraph.render("Lorem ipsum dolor sit amet, consectetur adipiscing elit.", width=80)

    def p_wrap() -> None:
        Paragraph.render(("Lorem ipsum " * 20).strip(), width=60)

    def prog() -> None:
        ProgressBar.render(72, 100, width=40, label="Load", color="green")

    def table() -> None:
        Table.render(["A", "B"], [["hello", 123], ["world", 456]], header_color="accent")

    for name, fn in (
        ("paragraph-small", p_small),
        ("paragraph-wrap", p_wrap),
        ("progress-bar", prog),
        ("table", table),
    ):
        results.append(_bench(name, fn))
    return results


def run_micro_suite() -> list[BenchResult]:
    """Run micro-benchmarks for core hot paths.

    Includes:
    - markup-parse: nested color and custom tags
    - gradient-gen: foreground gradient across a long string
    - layout-solve: flex row with wraps and gaps
    - vdom-diff: line diff with small edits across frames
    """
    results: list[BenchResult] = []

    # Markup parse
    engine = MarkupEngine()
    sample_text = (
        "[bold][fg=accent]Hello[/reset] {time:%H:%M:%S} "
        "[underline]world[/reset] and [#ff00ff]colors[/reset]!" * 5
    )
    results.append(
        _bench_call(
            "markup-parse",
            lambda: engine.render(sample_text),
            iters=500,
        )
    )

    # Gradient generation
    long = "The quick brown fox jumps over the lazy dog. " * 10
    results.append(
        _bench_call(
            "gradient-gen",
            lambda: Color.gradient(long, (78, 161, 255), (255, 176, 32)),
            iters=200,
        )
    )

    # Layout solve (row with wrapping)
    def _layout_case() -> None:
        root = LayoutNode(style=LayoutStyle(direction="row", wrap=True, gap=1, width=80, padding=1))
        for i in range(12):
            child = LayoutNode(style=LayoutStyle(width=6 + (i % 3), height=1))
            root.add(child)
        compute_layout(root)

    results.append(_bench_call("layout-solve", _layout_case, iters=300))

    # VDOM line diff on small edits
    base = [f"line {i}" for i in range(200)]

    def _diff_case() -> None:
        new = list(base)
        new[5] = "line 5 changed"
        new[123] = "line 123 changed"
        new.append("new tail")
        _ = diff_changes(base, new)

    results.append(_bench_call("vdom-diff", _diff_case, iters=500))

    return results


def measure_import_time_ms(python_executable: str | None = None) -> float:
    """Measure cold import time for ``textforge`` in a fresh interpreter.

    Uses a subprocess to avoid warm caches.
    """
    exe = python_executable or "python"
    code = (
        "import time; t=time.perf_counter(); import textforge; "
        "print((time.perf_counter()-t)*1000.0)"
    )
    try:
        out = subprocess.check_output([exe, "-c", code], stderr=subprocess.STDOUT, text=True)
        return float(out.strip().splitlines()[-1])
    except Exception:
        return -1.0


def measure_first_render_ms() -> float:
    """Measure the latency of the first simple render to a dummy stream."""
    class _Dummy(TextIO):
        buffer: io.StringIO

        def __init__(self) -> None:
            self.buffer = io.StringIO()
            super().__init__()

        def write(self, s: str) -> int:
            return self.buffer.write(s)

        def flush(self) -> None:
            pass

        @property
        def encoding(self) -> str:
            return "utf-8"

    c = Console(stream=_Dummy())
    start = time.perf_counter()
    c.print("Hello, world!", markup=False)
    return (time.perf_counter() - start) * 1000.0


def measure_live_update_ms(frames: int = 120) -> float:
    """Measure steady-state LiveSession update cost on a dummy stream.

    Returns average milliseconds per frame across ``frames`` updates.
    """
    class _Dummy(TextIO):
        buffer: io.StringIO

        def __init__(self) -> None:
            self.buffer = io.StringIO()
            super().__init__()

        def write(self, s: str) -> int:
            return self.buffer.write(s)

        def flush(self) -> None:
            pass

        @property
        def encoding(self) -> str:
            return "utf-8"

    c = Console(stream=_Dummy())
    with c.live() as live:
        text = "\n".join([f"row {i}" for i in range(50)])
        live.update(text)
        start = time.perf_counter()
        for i in range(frames):
            # Change a couple of lines to exercise small diffs
            changed = text.replace("row 10", f"row 10 [{i}]").replace("row 30", f"row 30 [{i}]")
            live.update(changed)
        total_ms = (time.perf_counter() - start) * 1000.0
        return total_ms / max(1, frames)
