"""Accessibility helpers and semantic metadata utilities for Textforge."""
from __future__ import annotations

from dataclasses import dataclass, field
from typing import TYPE_CHECKING, Any

from ..style.colors import Color

if TYPE_CHECKING:
    from collections.abc import Iterable


def parse_hex_to_rgb(spec: str) -> tuple[int, int, int] | None:
    """Backward-compatible helper that parses ``#RRGGBB`` strings.

    Prefer ``Color.resolve_rgb`` for general specs.
    """
    if not spec.startswith("#"):
        return None
    return Color.hex_to_rgb(spec)


def is_contrast_sufficient(fg: str, bg: str = "#000000", large_text: bool = False) -> bool:
    return Color.is_contrast_sufficient(fg, bg, large_text=large_text)


def simulate_color_blind(rgb: tuple[int, int, int], mode: str = "protanopia") -> tuple[int, int, int]:
    return Color.simulate_color_blind(rgb, mode)


def ensure_min_contrast(fg: str, bg: str, min_ratio: float = 4.5) -> str:
    return Color.ensure_min_contrast(fg, bg, min_ratio=min_ratio)


@dataclass
class AccessibilityNode:
    """Semantic metadata that can be attached to a renderable element."""

    role: str
    name: str
    description: str | None = None
    state: dict[str, Any] = field(default_factory=dict)
    shortcuts: list[str] = field(default_factory=list)
    children: list[AccessibilityNode] = field(default_factory=list)

    def add_child(self, child: AccessibilityNode) -> None:
        self.children.append(child)

    def aria_attributes(self) -> dict[str, str]:
        """Return a mapping suitable for ARIA attributes."""
        attrs: dict[str, str] = {"role": self.role, "aria-label": self.name}
        if self.description:
            attrs["aria-description"] = self.description
        if self.shortcuts:
            attrs["aria-keyshortcuts"] = ",".join(self.shortcuts)
        for key, value in self.state.items():
            aria_key = key.replace("_", "-")
            attrs[f"aria-{aria_key}"] = str(value)
        return attrs

    def to_dict(self) -> dict[str, Any]:
        """Serialize the node and its children into a plain dictionary."""
        return {
            "role": self.role,
            "name": self.name,
            "description": self.description,
            "state": dict(self.state),
            "shortcuts": list(self.shortcuts),
            "children": [child.to_dict() for child in self.children],
        }

    def summarize(self, indent: int = 0) -> str:
        """Return a human readable summary for debugging."""
        prefix = " " * indent
        parts = [f"{prefix}- {self.role}: {self.name}"]
        if self.description:
            parts.append(f"{prefix}  description: {self.description}")
        if self.state:
            parts.append(f"{prefix}  state: {self.state}")
        if self.shortcuts:
            parts.append(f"{prefix}  shortcuts: {', '.join(self.shortcuts)}")
        for child in self.children:
            parts.append(child.summarize(indent + 2))
        return "\n".join(parts)


class AccessibilityAPI:
    """High-level accessibility helpers and registry of semantic nodes."""

    _registry: dict[str, AccessibilityNode] = {}

    @staticmethod
    def is_contrast_ok(fg_hex: str, bg_hex: str, large_text: bool = False) -> bool:
        return is_contrast_sufficient(fg_hex, bg_hex, large_text=large_text)

    @staticmethod
    def suggest_contrast_safe_fg(fg_hex: str, bg_hex: str, min_ratio: float = 4.5) -> str:
        return ensure_min_contrast(fg_hex, bg_hex, min_ratio=min_ratio)

    @classmethod
    def register_node(
        cls,
        node_id: str,
        *,
        role: str,
        name: str,
        description: str | None = None,
        state: dict[str, Any] | None = None,
        shortcuts: Iterable[str] | None = None,
        children: Iterable[AccessibilityNode] | None = None,
    ) -> AccessibilityNode:
        """Register a semantic node for later inspection or export."""
        node = AccessibilityNode(
            role=role,
            name=name,
            description=description,
            state=dict(state or {}),
            shortcuts=list(shortcuts or []),
            children=list(children or []),
        )
        cls._registry[node_id] = node
        return node

    @classmethod
    def unregister_node(cls, node_id: str) -> None:
        cls._registry.pop(node_id, None)

    @classmethod
    def get_node(cls, node_id: str) -> AccessibilityNode | None:
        return cls._registry.get(node_id)

    @classmethod
    def annotate_progress(
        cls,
        node_id: str,
        *,
        label: str,
        current: float,
        maximum: float,
        unit: str | None = None,
    ) -> AccessibilityNode:
        """Create or update a node representing a progress indicator."""
        percentage = 0.0 if maximum <= 0 else min(max(current / maximum * 100.0, 0.0), 100.0)
        description = f"{current:.1f} of {maximum:.1f}"
        if unit:
            description += f" {unit}"
        description += f" ({percentage:.0f} percent)"
        node = cls.register_node(
            node_id,
            role="progressbar",
            name=label,
            description=description,
            state={"valuenow": f"{current:.1f}", "valuemax": f"{maximum:.1f}"},
        )
        return node

    @classmethod
    def compose_alt_text(
        cls,
        label: str,
        lines: Iterable[str],
        *,
        max_lines: int = 5,
    ) -> str:
        """Compose condensed alt text for rich renderables."""
        kept: list[str] = []
        for idx, line in enumerate(lines):
            if idx >= max_lines:
                kept.append("...")
                break
            kept.append(line.strip())
        snippet = " ".join(part for part in kept if part)
        return f"{label}: {snippet}"

    @classmethod
    def export_registry(cls) -> dict[str, dict[str, Any]]:
        """Return a JSON-serialisable representation of the registry."""
        return {key: node.to_dict() for key, node in cls._registry.items()}

    @classmethod
    def describe_registry(cls) -> str:
        """Return a human-friendly multi-line summary of registered nodes."""
        if not cls._registry:
            return "No accessibility metadata registered."
        lines: list[str] = []
        for key, node in cls._registry.items():
            lines.append(f"[{key}]")
            lines.append(node.summarize(indent=2))
        return "\n".join(lines)

    @classmethod
    def clear(cls) -> None:
        cls._registry.clear()
