"""Comprehensive showcase demo rendering most components."""

from __future__ import annotations

from ..components import (
    Accordion,
    AsciiArt,
    Banner,
    BlockQuote,
    Box,
    Breadcrumbs,
    Card,
    Chart,
    Checkbox,
    Columns,
    Dialog,
    Divider,
    GraphCanvas,
    Grid,
    HorizontalMenu,
    ListElement,
    Menu,
    Meter,
    Panel,
    Paragraph,
    ProgressBar,
    Radio,
    SegmentedBar,
    Slider,
    Spectrogram,
    Spinner,
    Table,
    Tabs,
    Title,
    Tooltip,
    Waveform,
)
from ..components import data as data_components
from ..components import feedback as feedback_components
from ..components.navigation import FileBrowser
from ..core import composite, render_call, tfprint


class Showcase:
    @staticmethod
    def render_full(width: int = 80):
        import math
        return composite(
            [
                render_call(Banner.render, "TEXTFORGE SHOWCASE", style="double", width=width, color="accent"),
                render_call(Title.render, "Typography", width=width, symbol_style="rounded", color="primary"),
                render_call(Paragraph.render, "[fg=muted]Textforge renders various components with simple markup and themes.[reset]", width=width),
                render_call(ListElement.render, ["Bullet one", "Bullet two", "[accent]Colored[reset] bullet three"], width=width),
                render_call(BlockQuote.render, "Simplicity is the ultimate sophistication.", author="Leonardo da Vinci", width=width),
                render_call(Divider.render, width=width, style="heavy", text="Status"),
                render_call(ProgressBar.render, 72, 100, label="Loading", color="success"),
                render_call(Meter.render, 64.3, label="CPU", thresholds={0.0: "green", 0.6: "yellow", 0.8: "red"}),
                render_call(SegmentedBar.render, 7, 10, label="Energy", color="accent"),
                render_call(print, Spinner.render("Thinking", frame=5)),
                render_call(Divider.render, width=width, style="solid", text="Data"),
                render_call(Table.render, ["Name", "Score"], [["Alice", 91], ["Bob", 77], ["Carol", 85]], header_color="primary"),
                render_call(Chart.bar_chart, {"A": 5, "B": 8, "C": 3}, width=width, max_bar_width=40),
                render_call(Chart.sparkline, [1, 5, 3, 6, 2, 7, 4]),
                render_call(Chart.area_chart, [1, 2, 3, 2, 4, 3, 5, 4, 6, 3], width=width, height=6),
                render_call(Chart.histogram, [1, 1, 2, 3, 5, 8, 13, 21, 21, 21], bins=5),
                render_call(Divider.render, width=width, style="dashed", text="Containers"),
                render_call(Box.render, "Simple boxed text", width=width),
                render_call(Card.render, "Player", {"HP": 90, "MP": 40, "Gold": 123}, width=width),
                render_call(
                    Panel.render,
                    [
                        {"title": "Left", "content": ["One", "Two", "Three"]},
                        {"title": "Right", "content": ["Alpha", "Beta"]},
                    ],
                    width=width,
                ),
                render_call(Divider.render, width=width, style="solid", text="Layout"),
                render_call(Columns.render, [["Left A", "Left B"], ["Right A", "Right B"]]),
                render_call(Grid.render, [["Cell 1", "Cell 2"], ["Cell 3", "Cell 4"]], cell_width=20, cell_height=2),
                render_call(Divider.render, width=width, style="solid", text="Navigation"),
                render_call(Breadcrumbs.render, ["Home", "Library", "Data"]),
                render_call(Menu.render, "Main Menu", ["Start", "Options", "Exit"], selected_index=1),
                render_call(HorizontalMenu.render, ["Tab1", "Tab2", "Tab3"], selected_index=2),
                render_call(Tabs.render, ["Overview", "Details", "Settings"], active_index=1),
                render_call(
                    Accordion.render,
                    [
                        {"title": "Section One", "content": "First section content"},
                        {"title": "Section Two", "content": "Second section content"},
                    ],
                    expanded_index=0,
                ),
                render_call(Divider.render, width=width, style="solid", text="Feedback & Code"),
                render_call(print, feedback_components.Badge.render("BETA", style="round", color="warning")),
                render_call(print, feedback_components.Rating.render(4.2, style="stars")),
                render_call(data_components.CodeBlock.render, "def hello():\n    return 'world'", language="python", width=width),
                render_call(feedback_components.Toast.render, "This is a toast message", tone="info", width=width),
                render_call(feedback_components.ConsolePanel.render, ["Log line 1", "[warning]Warning line[reset]", "Info line"], width=width),
                render_call(Tooltip.render, "Helpful tip here", width=40),
                render_call(feedback_components.Snackbar.render, "Saved", tone="success"),
                render_call(feedback_components.AlertBanner.render, "System maintenance at 2am", tone="warning", width=width),
                render_call(Divider.render, width=width, style="double", text="Dialogs"),
                render_call(Dialog.render, "Confirm", "Do you want to continue?", buttons=["Yes", "No"], selected_button=0, width=width),
                render_call(Divider.render, width=width, style="dotted", text="Interactive (static preview)"),
                render_call(Checkbox.render, "Enable feature X", checked=True),
                render_call(Radio.render, "Choice A", selected=True),
                render_call(Slider.render, "Volume", value=75, width=30),
                render_call(Divider.render, width=width, style="decorative", text="ASCII Art"),
                render_call(AsciiArt.render, "TF", font="tiny", color="accent"),
                render_call(Divider.render, width=width, style="solid", text="Visualizers"),
                render_call(GraphCanvas.render, lambda x: math.sin(x)),
                render_call(Waveform.render, [math.sin(i / 3.0) for i in range(60)]),
                render_call(Spectrogram.render, [[i * j % 10 for i in range(40)] for j in range(10)]),
                render_call(Divider.render, width=width, style="solid", text="File Browser (first 10)"),
                render_call(FileBrowser.render, path=".", max_entries=10),
            ],
            separator="\n"
        )

    @staticmethod
    def full(width: int = 80) -> None:
        tfprint(Showcase.render_full(width))
