"""Game-oriented template helpers composed from base components."""
from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Literal

from ..components.containers import Box, Card
from ..components.decorative import Banner, Divider
from ..components.status import ProgressBar
from ..components.typography import ListElement, Paragraph, Title
from ..core import composite, render_call, tfprint
from ..style.colors import Color
from ..utils import format_text


@dataclass(slots=True)
class GameHeaderConfig:
    title: str
    subtitle: str = ""
    width: int = 80


class Template:
    """Pre-built UI component templates."""

    @staticmethod
    def render_game_header(cfg: GameHeaderConfig):
        parts = [render_call(Banner.render, cfg.title, style="double", width=cfg.width, color="gold", blank_lines_before=1)]
        if cfg.subtitle:
            parts.append(render_call(format_text, cfg.subtitle, width=cfg.width, blank_lines=(0, 1)))
        return composite(parts)

    @staticmethod
    def game_header(title: str, subtitle: str = "", width: int = 80) -> None:
        tfprint(Template.render_game_header(GameHeaderConfig(title=title, subtitle=subtitle, width=width)))

    @staticmethod
    def render_combat_status(
        player_hp: int,
        player_max_hp: int,
        enemy_name: str,
        enemy_hp: int,
        enemy_max_hp: int,
        width: int = 80,
    ):
        parts = [render_call(Divider.render, width=width, style="heavy", text="COMBAT", color="red")]
        parts.append(render_call(print, "\n" + Color.BOLD + Color.NAMED_COLORS["green"] + "PLAYER" + Color.RESET))
        parts.append(render_call(ProgressBar.render, player_hp, player_max_hp, width=40, label="HP", color="green"))
        parts.append(render_call(print))
        parts.append(render_call(print, Color.BOLD + Color.NAMED_COLORS["red"] + "ENEMY: " + enemy_name + Color.RESET))
        parts.append(render_call(ProgressBar.render, enemy_hp, enemy_max_hp, width=40, label="HP", color="red"))
        parts.append(render_call(Divider.render, width=width, style="solid", color="gray", blank_lines_before=1))
        return composite(parts)

    @staticmethod
    def combat_status(
        player_hp: int,
        player_max_hp: int,
        enemy_name: str,
        enemy_hp: int,
        enemy_max_hp: int,
        width: int = 80,
    ) -> None:
        tfprint(Template.render_combat_status(player_hp, player_max_hp, enemy_name, enemy_hp, enemy_max_hp, width))

    @staticmethod
    def render_inventory_screen(items: list[dict[str, Any]], width: int = 80):
        parts = [render_call(Title.render, "INVENTORY", width=width, symbol_style="double", color="cyan")]
        if not items:
            parts.append(render_call(Paragraph.render, "Your inventory is empty.", width=width, indent=2, color="gray"))
        else:
            item_list: list[str] = []
            for item in items:
                name = item.get("name", "Unknown")
                qty = item.get("quantity", 1)
                rarity = item.get("rarity", "common")
                rarity_colors = {
                    "common": "white",
                    "uncommon": "green",
                    "rare": "blue",
                    "epic": "purple",
                    "legendary": "gold",
                }
                color = rarity_colors.get(rarity, "white")
                item_list.append(f"[{color}]{name}[reset] x{qty}")
            parts.append(render_call(ListElement.render, item_list, width=width, list_type="bullet", marker_color="cyan"))
        return composite(parts)

    @staticmethod
    def inventory_screen(items: list[dict[str, Any]], width: int = 80) -> None:
        tfprint(Template.render_inventory_screen(items, width))

    @staticmethod
    def render_character_sheet(name: str, level: int, stats: dict[str, int], width: int = 60):
        return render_call(
            Card.render,
            title=f"{name} - Level {level}",
            content=stats,
            width=width,
            title_color="gold",
            border_style="double",
        )

    @staticmethod
    def character_sheet(name: str, level: int, stats: dict[str, int], width: int = 60) -> None:
        tfprint(Template.render_character_sheet(name, level, stats, width))

    @staticmethod
    def render_notification(message: str, notification_type: Literal["info", "success", "warning", "error"] = "info", width: int = 60):
        icons = {"info": "ℹ", "success": "✓", "warning": "⚠", "error": "✗"}
        colors = {"info": "cyan", "success": "green", "warning": "yellow", "error": "red"}
        icon = icons.get(notification_type, "ℹ")
        color = colors.get(notification_type, "cyan")
        return render_call(
            Box.render,
            f"{icon} {message}",
            width=width,
            color=color,
            symbol_style="heavy",
            padding=1,
            blank_lines_before=1,
            blank_lines_after=1,
        )

    @staticmethod
    def notification(message: str, notification_type: Literal["info", "success", "warning", "error"] = "info", width: int = 60) -> None:
        tfprint(Template.render_notification(message, notification_type, width))

    @staticmethod
    def render_loading_screen(message: str = "Loading", progress: int = 0, total: int = 100, width: int = 60):
        return composite(
            [
                render_call(print, "\n\n"),
                render_call(format_text, message, width=width, color="cyan"),
                render_call(
                    ProgressBar.render,
                    progress,
                    total,
                    width=50,
                    show_percentage=True,
                    show_numbers=False,
                    color="cyan",
                    blank_lines_before=1,
                    blank_lines_after=2,
                ),
            ]
        )

    @staticmethod
    def loading_screen(message: str = "Loading", progress: int = 0, total: int = 100, width: int = 60) -> None:
        tfprint(Template.render_loading_screen(message, progress, total, width))
