"""Business templates: reports, invoices, KPI summaries."""

from __future__ import annotations

from dataclasses import dataclass

from ..components import key_value, paragraph, table, title
from ..components.containers import Box
from ..components.decorative import Divider
from ..core import composite, render_call, tfprint


@dataclass(slots=True)
class ReportCardConfig:
    title: str
    items: dict[str, str]
    width: int = 80
    title_style: str = "double"
    title_color: str = "primary"
    indent: int = 2


@dataclass(slots=True)
class InvoiceConfig:
    company: str
    invoice_no: str
    rows: list[tuple[str, int, float]]
    width: int = 80
    header_style: str = "rounded"
    header_color: str = "accent"
    table_header_color: str = "primary"
    total_box_style: str = "simple"


@dataclass(slots=True)
class KpiSummaryConfig:
    stats: dict[str, float]
    width: int = 80
    title_style: str = "rounded"
    title_color: str = "success"


class Business:
    @staticmethod
    def render_report_card(cfg: ReportCardConfig):
        return composite(
            [
                title(cfg.title, width=cfg.width, symbol_style=cfg.title_style, color=cfg.title_color),
                key_value(cfg.items, indent=cfg.indent),
                render_call(Divider.render, width=cfg.width),
            ]
        )

    @staticmethod
    def report_card(title: str, items: dict[str, str], width: int = 80) -> None:
        tfprint(Business.render_report_card(ReportCardConfig(title=title, items=items, width=width)))

    @staticmethod
    def render_invoice(cfg: InvoiceConfig):
        table_rows = [[name, qty, f"${price:.2f}", f"${qty * price:.2f}"] for (name, qty, price) in cfg.rows]
        total = sum(q * p for (_n, q, p) in cfg.rows)
        return composite(
            [
                title("INVOICE", width=cfg.width, symbol_style=cfg.header_style, color=cfg.header_color),
                paragraph(f"Company: {cfg.company}\nInvoice #: {cfg.invoice_no}", width=cfg.width),
                render_call(Divider.render, width=cfg.width, style="dashed"),
                table(["Item", "Qty", "Price", "Total"], table_rows, header_color=cfg.table_header_color),
                render_call(
                    Box.render,
                    f"Total Due: [bold][gold]${total:.2f}[reset]",
                    width=cfg.width,
                    symbol_style=cfg.total_box_style,
                ),
            ]
        )

    @staticmethod
    def invoice(company: str, invoice_no: str, rows: list[tuple[str, int, float]], width: int = 80) -> None:
        tfprint(Business.render_invoice(InvoiceConfig(company=company, invoice_no=invoice_no, rows=rows, width=width)))

    @staticmethod
    def render_kpi_summary(cfg: KpiSummaryConfig):
        rows = [[k, f"{v:.2f}"] for k, v in cfg.stats.items()]
        return composite(
            [
                title("KPI SUMMARY", width=cfg.width, symbol_style=cfg.title_style, color=cfg.title_color),
                table(["Metric", "Value"], rows, header_color=cfg.title_color),
            ]
        )

    @staticmethod
    def kpi_summary(stats: dict[str, float], width: int = 80) -> None:
        tfprint(Business.render_kpi_summary(KpiSummaryConfig(stats=stats, width=width)))
