"""Symbol presets and glyph registries for Textforge.

Supports both Unicode and ASCII symbol modes for maximum compatibility.
"""
from __future__ import annotations

import os
import sys
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from collections.abc import Mapping


class UnicodeSymbolRegistry:
    """Unicode symbol registry with comprehensive glyph mappings."""

    PRESETS: dict[str, dict[str, str]] = {
        "box": {
            "top_left": "┌",
            "top_right": "┐",
            "bottom_left": "└",
            "bottom_right": "┘",
            "horizontal": "─",
            "vertical": "│",
            "cross": "┼",
            "t_down": "┬",
            "t_up": "┴",
            "t_right": "├",
            "t_left": "┤",
        },
        "double": {
            "top_left": "╔",
            "top_right": "╗",
            "bottom_left": "╚",
            "bottom_right": "╝",
            "horizontal": "═",
            "vertical": "║",
            "cross": "╬",
            "t_down": "╦",
            "t_up": "╩",
            "t_right": "╠",
            "t_left": "╣",
        },
        "rounded": {
            "top_left": "╭",
            "top_right": "╮",
            "bottom_left": "╰",
            "bottom_right": "╯",
            "horizontal": "─",
            "vertical": "│",
            "cross": "┼",
            "t_down": "┬",
            "t_up": "┴",
            "t_right": "├",
            "t_left": "┤",
        },
        "heavy": {
            "top_left": "┏",
            "top_right": "┓",
            "bottom_left": "┗",
            "bottom_right": "┛",
            "horizontal": "━",
            "vertical": "┃",
            "cross": "╋",
            "t_down": "┳",
            "t_up": "┻",
            "t_right": "┣",
            "t_left": "┫",
        },
        "dashed": {
            "top_left": "┌",
            "top_right": "┐",
            "bottom_left": "└",
            "bottom_right": "┘",
            "horizontal": "╌",
            "vertical": "╎",
            "cross": "┼",
            "t_down": "┬",
            "t_up": "┴",
            "t_right": "├",
            "t_left": "┤",
        },
        "stars": {
            "top_left": "✶",
            "top_right": "✶",
            "bottom_left": "✶",
            "bottom_right": "✶",
            "horizontal": "✶",
            "vertical": "✶",
            "cross": "✶",
            "t_down": "✶",
            "t_up": "✶",
            "t_right": "✶",
            "t_left": "✶",
        },
        "arrows": {
            "top_left": "↖",
            "top_right": "↗",
            "bottom_left": "↙",
            "bottom_right": "↘",
            "horizontal": "→",
            "vertical": "↓",
            "cross": "✚",
            "t_down": "↓",
            "t_up": "↑",
            "t_right": "→",
            "t_left": "←",
        },
        "diamond": {
            "top_left": "◇",
            "top_right": "◇",
            "bottom_left": "◇",
            "bottom_right": "◇",
            "horizontal": "◆",
            "vertical": "◆",
            "cross": "◆",
            "t_down": "▼",
            "t_up": "▲",
            "t_right": "▶",
            "t_left": "◀",
        },
        "thick": {
            "top_left": "█",
            "top_right": "█",
            "bottom_left": "█",
            "bottom_right": "█",
            "horizontal": "█",
            "vertical": "█",
            "cross": "█",
            "t_down": "█",
            "t_up": "█",
            "t_right": "█",
            "t_left": "█",
        },
        "dots": {
            "top_left": "∙",
            "top_right": "∙",
            "bottom_left": "∙",
            "bottom_right": "∙",
            "horizontal": "·",
            "vertical": "·",
            "cross": "·",
            "t_down": "·",
            "t_up": "·",
            "t_right": "·",
            "t_left": "·",
        },
        "math": {
            "plus": "+",
            "minus": "−",
            "times": "×",
            "divide": "÷",
            "equals": "=",
            "not_equals": "≠",
            "approx": "≈",
            "identical": "≡",
            "proportional": "∝",
            "less": "＜",
            "greater": "＞",
            "leq": "≤",
            "geq": "≥",
            "much_less": "≪",
            "much_greater": "≫",
            "plus_minus": "±",
            "minus_plus": "∓",
            "infinity": "∞",
            "integral": "∫",
            "double_integral": "∬",
            "sum": "∑",
            "product": "∏",
            "sqrt": "√",
            "cube_root": "∛",
            "fourth_root": "∜",
            "partial": "∂",
            "angle": "∠",
            "perpendicular": "⟂",
            "parallel": "∥",
            "not_parallel": "∦",
            "element_of": "∈",
            "not_element_of": "∉",
            "subset": "⊂",
            "superset": "⊃",
            "subset_eq": "⊆",
            "superset_eq": "⊇",
            "intersection": "∩",
            "union": "∪",
            "logical_and": "∧",
            "logical_or": "∨",
            "forall": "∀",
            "exists": "∃",
            "therefore": "∴",
            "because": "∵",
        },
        "geometry": {
            "circle_black": "●",
            "circle_white": "○",
            "circle_medium": "◉",
            "circle_bullseye": "◎",
            "square_black": "■",
            "square_white": "□",
            "square_round": "◻",
            "rectangle_black": "▬",
            "rectangle_white": "▭",
            "diamond_black": "◆",
            "diamond_white": "◇",
            "lozenge": "◊",
            "triangle_up": "▲",
            "triangle_down": "▼",
            "triangle_left": "◀",
            "triangle_right": "▶",
            "triangle_up_white": "△",
            "triangle_down_white": "▽",
            "parallelogram_white": "▱",
            "parallelogram_black": "▰",
            "pentagon_black": "⬟",
            "pentagon_white": "⬠",
            "hexagon_black": "⬢",
            "hexagon_white": "⬡",
            "octagon_black": "⯃",
            "octagon_white": "⯁",
        },
        "box2": {
            "light_h": "─",
            "light_v": "│",
            "light_t_up": "┴",
            "light_t_down": "┬",
            "light_t_left": "┤",
            "light_t_right": "├",
            "light_cross": "┼",
            "light_ul": "┌",
            "light_ur": "┐",
            "light_ll": "└",
            "light_lr": "┘",
            "double_h": "═",
            "double_v": "║",
            "double_cross": "╬",
            "double_ul": "╔",
            "double_ur": "╗",
            "double_ll": "╚",
            "double_lr": "╝",
            "heavy_h": "━",
            "heavy_v": "┃",
            "heavy_cross": "╋",
            "heavy_ul": "┏",
            "heavy_ur": "┓",
            "heavy_ll": "┗",
            "heavy_lr": "┛",
        },
        "bullets": {
            "bullet": "•",
            "bullet_small": "∙",
            "bullet_triangle": "‣",
            "dash": "–",
            "emdash": "—",
            "middot": "·",
            "asterisk": "＊",
            "section": "§",
            "paragraph": "¶",
        },
        "currency": {
            "dollar": "＄",
            "euro": "€",
            "pound": "£",
            "yen": "¥",
            "rupee": "₹",
            "ruble": "₽",
            "won": "₩",
            "naira": "₦",
            "hryvnia": "₴",
            "bitcoin": "₿",
            "cent": "¢",
        },
        "weather": {
            "sun": "☀",
            "cloud": "☁",
            "umbrella": "☂",
            "snowman": "☃",
            "comet": "☄",
            "star_six": "✶",
            "sparkles": "✨",
            "lightning": "⚡",
            "snowflake": "❄",
            "thermometer": "🌡",
        },
        "cards": {
            "heart_black": "♥",
            "diamond_black": "♦",
            "club_black": "♣",
            "spade_black": "♠",
            "heart_white": "♡",
            "diamond_white": "♢",
            "club_white": "♧",
            "spade_white": "♤",
        },
        "music": {
            "note_eighth": "♪",
            "note_beamed": "♫",
            "sharp": "♯",
            "flat": "♭",
            "natural": "♮",
            "rest_quarter": "𝄽",
        },
        "greek": {
            "alpha": "α",
            "beta": "β",
            "gamma": "γ",
            "delta": "δ",
            "epsilon": "ε",
            "theta": "θ",
            "lambda": "λ",
            "mu": "μ",
            "pi": "π",
            "sigma": "σ",
            "phi": "φ",
            "omega": "ω",
            "Gamma": "Γ",
            "Delta": "Δ",
            "Theta": "Θ",
            "Lambda": "Λ",
            "Pi": "Π",
            "Sigma": "Σ",
            "Phi": "Φ",
            "Omega": "Ω",
        },
        "chevrons": {
            "chev_left": "‹",
            "chev_right": "›",
            "chev_left_double": "«",
            "chev_right_double": "»",
            "caret_up": "︿",
            "caret_down": "﹀",
        },
        "blocks": {
            "shade_light": "░",
            "shade_medium": "▒",
            "shade_dark": "▓",
            "block_full": "█",
            "block_upper": "▀",
            "block_lower": "▄",
            "block_left": "▌",
            "block_right": "▐",
        },
    }

    BAR_FILLED = "█"
    BAR_EMPTY = "░"
    BAR_PARTIAL = ["▓", "▒", "░", " "]

    ARROWS: Mapping[str, str] = {
        "up": "↑",
        "down": "↓",
        "left": "←",
        "right": "→",
        "up_down": "↕",
        "left_right": "↔",
        "upleft": "↖",
        "upright": "↗",
        "downleft": "↙",
        "downright": "↘",
        "thick_right": "➜",
        "thick_left": "⬅",
        "thick_up": "⬆",
        "thick_down": "⬇",
        "long_right": "⟶",
        "long_left": "⟵",
        "long_up": "⟰",
        "long_down": "⟱",
        "hook_right": "↪",
        "hook_left": "↩",
        "curve_right": "⤷",
        "curve_left": "⤶",
        "barbed_right": "➤",
        "barbed_left": "◀",
    }

    SHAPES: Mapping[str, str] = {
        "circle": "●",
        "circle_empty": "○",
        "square": "■",
        "square_empty": "□",
        "diamond": "◆",
        "diamond_empty": "◇",
        "triangle": "▲",
        "triangle_empty": "△",
        "circle_medium": "◉",
        "bullseye": "◎",
        "ellipse_h": "⬭",
        "ellipse_v": "⬯",
        "hexagon": "⬢",
        "hexagon_empty": "⬡",
        "pentagon": "⬟",
        "pentagon_empty": "⬠",
        "octagon": "⯃",
        "octagon_empty": "⯁",
        "lozenge": "◊",
        "parallelogram": "▰",
        "parallelogram_empty": "▱",
    }

    STATUS: Mapping[str, str] = {
        "check": "✔",
        "cross": "✖",
        "warning": "⚠",
        "info": "ℹ",
        "star": "★",
        "star_empty": "☆",
        "heart": "❤",
        "heart_empty": "♡",
        "skull": "☠",
        "sword": "🗡",
        "shield": "🛡",
        "potion": "🧪",
        "coin": "🪙",
        "gem": "💎",
        "question": "❓",
        "exclamation": "❗",
        "pause": "⏸",
        "play": "▶",
        "stop": "⏹",
        "record": "⏺",
        "hourglass": "⏳",
        "success": "✅",
        "failure": "❌",
        "pending": "⏲",
        "lock": "🔒",
        "unlock": "🔓",
        "battery": "🔋",
        "trash": "🗑",
        "search": "🔎",
        "gear": "⚙",
    }

    TREE: Mapping[str, str] = {
        "branch": "├── ",
        "branch_last": "└── ",
        "vertical": "│   ",
        "space": "    ",
        "branch_mid": "├─ ",
        "branch_heavy": "┣━ ",
        "branch_last_heavy": "┗━ ",
        "vertical_heavy": "┃   ",
    }

    LINES: Mapping[str, str] = {
        "solid": "─",
        "double": "═",
        "dashed": "┄",
        "dotted": "·",
        "heavy": "━",
        "decorative": "═══",
        "wavy": "﹏",
        "hair": "⎯",
        "thick": "▇",
        "thin": "⎼",
        "triple": "┅",
    }

    BLOCK_CHART: str = "▁▂▃▄▅▆▇█"

    UI_ICONS: Mapping[str, str] = {
        "folder": "📁",
        "file": "📄",
        "arrow_right": "►",
        "checkbox_checked": "☑",
        "checkbox_unchecked": "☐",
        "radio_selected": "◉",
        "radio_unselected": "◯",
        "quote": "┃",
        "checkmark": "✓",
        "home": "🏠",
        "settings": "⚙",
        "download": "⭳",
        "upload": "⭡",
        "link": "🔗",
        "unlink": "❌",
        "terminal": "🖥",
        "keyboard": "⌨",
        "mouse": "🖱",
        "clipboard": "📋",
        "pin": "📌",
        "bell": "🔔",
        "bell_off": "🔕",
        "eye": "👁",
        "eye_off": "🙈",
    }

    SEPARATORS: Mapping[str, str] = {
        "arrow": "›",
        "dash": "—",
        "pipe": "│",
        "double_pipe": "║",
        "middot": "·",
        "bull": "•",
        "chevron": "»",
        "slash": "/",
        "backslash": "\\",
        "tilde": "~",
    }


class ASCIISymbolRegistry:
    """ASCII symbol registry with 1:1 mappings for maximum compatibility."""

    PRESETS: dict[str, dict[str, str]] = {
        "box": {
            "top_left": "+",
            "top_right": "+",
            "bottom_left": "+",
            "bottom_right": "+",
            "horizontal": "-",
            "vertical": "|",
            "cross": "+",
            "t_down": "+",
            "t_up": "+",
            "t_right": "+",
            "t_left": "+",
        },
        "double": {
            "top_left": "#",
            "top_right": "#",
            "bottom_left": "#",
            "bottom_right": "#",
            "horizontal": "=",
            "vertical": "#",
            "cross": "#",
            "t_down": "#",
            "t_up": "#",
            "t_right": "#",
            "t_left": "#",
        },
        "rounded": {
            "top_left": "/",
            "top_right": "\\",
            "bottom_left": "\\",
            "bottom_right": "/",
            "horizontal": "-",
            "vertical": "|",
            "cross": "+",
            "t_down": "+",
            "t_up": "+",
            "t_right": "+",
            "t_left": "+",
        },
        "heavy": {
            "top_left": "#",
            "top_right": "#",
            "bottom_left": "#",
            "bottom_right": "#",
            "horizontal": "=",
            "vertical": "#",
            "cross": "#",
            "t_down": "#",
            "t_up": "#",
            "t_right": "#",
            "t_left": "#",
        },
        "dashed": {
            "top_left": "+",
            "top_right": "+",
            "bottom_left": "+",
            "bottom_right": "+",
            "horizontal": "~",
            "vertical": ":",
            "cross": "+",
            "t_down": "+",
            "t_up": "+",
            "t_right": "+",
            "t_left": "+",
        },
        "stars": {
            "top_left": "*",
            "top_right": "*",
            "bottom_left": "*",
            "bottom_right": "*",
            "horizontal": "*",
            "vertical": "*",
            "cross": "*",
            "t_down": "*",
            "t_up": "*",
            "t_right": "*",
            "t_left": "*",
        },
        "arrows": {
            "top_left": "^",
            "top_right": "^",
            "bottom_left": "v",
            "bottom_right": "v",
            "horizontal": ">",
            "vertical": "v",
            "cross": "+",
            "t_down": "v",
            "t_up": "^",
            "t_right": ">",
            "t_left": "<",
        },
        "diamond": {
            "top_left": "<",
            "top_right": ">",
            "bottom_left": "<",
            "bottom_right": ">",
            "horizontal": "<>",
            "vertical": "<>",
            "cross": "<>",
            "t_down": "v",
            "t_up": "^",
            "t_right": ">",
            "t_left": "<",
        },
        "thick": {
            "top_left": "#",
            "top_right": "#",
            "bottom_left": "#",
            "bottom_right": "#",
            "horizontal": "#",
            "vertical": "#",
            "cross": "#",
            "t_down": "#",
            "t_up": "#",
            "t_right": "#",
            "t_left": "#",
        },
        "dots": {
            "top_left": ".",
            "top_right": ".",
            "bottom_left": ".",
            "bottom_right": ".",
            "horizontal": ".",
            "vertical": ".",
            "cross": ".",
            "t_down": ".",
            "t_up": ".",
            "t_right": ".",
            "t_left": ".",
        },

        "math": {
            "plus": "+",
            "minus": "-",
            "times": "x",
            "divide": "/",
            "equals": "=",
            "not_equals": "!=",
            "approx": "~=",
            "identical": "==",
            "proportional": "~",
            "less": "<",
            "greater": ">",
            "leq": "<=",
            "geq": ">=",
            "much_less": "<<",
            "much_greater": ">>",
            "plus_minus": "+/-",
            "minus_plus": "-/+",
            "infinity": "oo",
            "integral": "S",
            "double_integral": "SS",
            "sum": "SUM",
            "product": "PROD",
            "sqrt": "sqrt",
            "cube_root": "cuberoot",
            "fourth_root": "root4",
            "partial": "d",
            "angle": "<>",
            "perpendicular": "T",
            "parallel": "||",
            "not_parallel": "!||",
            "element_of": "E",
            "not_element_of": "!E",
            "subset": "SUB",
            "superset": "SUP",
            "subset_eq": "SUB=",
            "superset_eq": "SUP=",
            "intersection": "INTER",
            "union": "UNION",
            "logical_and": "&&",
            "logical_or": "||",
            "forall": "A",
            "exists": "E?",
            "therefore": "::. ",
            "because": "':.",
        },
        "geometry": {
            "circle_black": "O",
            "circle_white": "o",
            "circle_medium": "(@)",
            "circle_bullseye": "(*)",
            "square_black": "#",
            "square_white": "[]",
            "square_round": "[ ]",
            "rectangle_black": "==",
            "rectangle_white": "[]",
            "diamond_black": "<>",
            "diamond_white": "< >",
            "lozenge": "<>",
            "triangle_up": "^",
            "triangle_down": "v",
            "triangle_left": "<",
            "triangle_right": ">",
            "triangle_up_white": "/\\",
            "triangle_down_white": "\\/",
            "parallelogram_white": "/__/",
            "parallelogram_black": "/##/",
            "pentagon_black": "(#5)",
            "pentagon_white": "( 5 )",
            "hexagon_black": "[#6]",
            "hexagon_white": "[ 6 ]",
            "octagon_black": "[#8]",
            "octagon_white": "[ 8 ]",
        },
        "box2": {
            "light_h": "-",
            "light_v": "|",
            "light_t_up": "+",
            "light_t_down": "+",
            "light_t_left": "+",
            "light_t_right": "+",
            "light_cross": "+",
            "light_ul": "+",
            "light_ur": "+",
            "light_ll": "+",
            "light_lr": "+",
            "double_h": "=",
            "double_v": "#",
            "double_cross": "#",
            "double_ul": "#",
            "double_ur": "#",
            "double_ll": "#",
            "double_lr": "#",
            "heavy_h": "=",
            "heavy_v": "#",
            "heavy_cross": "#",
            "heavy_ul": "#",
            "heavy_ur": "#",
            "heavy_ll": "#",
            "heavy_lr": "#",
        },
        "bullets": {
            "bullet": "*",
            "bullet_small": ".",
            "bullet_triangle": ">",
            "dash": "-",
            "emdash": "--",
            "middot": ".",
            "asterisk": "*",
            "section": "S",
            "paragraph": "P",
        },
        "currency": {
            "dollar": "$",
            "euro": "EUR",
            "pound": "GBP",
            "yen": "YEN",
            "rupee": "INR",
            "ruble": "RUB",
            "won": "WON",
            "naira": "NGN",
            "hryvnia": "UAH",
            "bitcoin": "BTC",
            "cent": "c",
        },
        "weather": {
            "sun": "*",
            "cloud": "(~)",
            "umbrella": "(U)",
            "snowman": "(snow)",
            "comet": "(*)->",
            "star_six": "*",
            "sparkles": "**",
            "lightning": "!",
            "snowflake": "*",
            "thermometer": "| |",
        },
        "cards": {
            "heart_black": "<3",
            "diamond_black": "<>",
            "club_black": "(+)",
            "spade_black": "^",
            "heart_white": "</3",
            "diamond_white": "< >",
            "club_white": "( )",
            "spade_white": "v",
        },
        "music": {
            "note_eighth": "*",
            "note_beamed": "**",
            "sharp": "#",
            "flat": "b",
            "natural": "n",
            "rest_quarter": "R",
        },
        "greek": {
            "alpha": "a",
            "beta": "b",
            "gamma": "g",
            "delta": "d",
            "epsilon": "e",
            "theta": "th",
            "lambda": "l",
            "mu": "m",
            "pi": "pi",
            "sigma": "s",
            "phi": "ph",
            "omega": "w",
            "Gamma": "G",
            "Delta": "D",
            "Theta": "Th",
            "Lambda": "L",
            "Pi": "Pi",
            "Sigma": "S",
            "Phi": "Ph",
            "Omega": "W",
        },
        "chevrons": {
            "chev_left": "<",
            "chev_right": ">",
            "chev_left_double": "<<",
            "chev_right_double": ">>",
            "caret_up": "^",
            "caret_down": "v",
        },
        "blocks": {
            "shade_light": ".",
            "shade_medium": "*",
            "shade_dark": "#",
            "block_full": "#",
            "block_upper": "^",
            "block_lower": "_",
            "block_left": "[",
            "block_right": "]",
        },
    }

    BAR_FILLED = "#"
    BAR_EMPTY = "."
    BAR_PARTIAL = ["#", "*", ".", " "]

    ARROWS: Mapping[str, str] = {
        "up": "^",
        "down": "v",
        "left": "<",
        "right": ">",
        "up_down": "|",
        "left_right": "-",
        "upleft": "<^",
        "upright": "^>",
        "downleft": "<v",
        "downright": "v>",
        "thick_right": ">>",
        "thick_left": "<<",
        "thick_up": "^^",
        "thick_down": "vv",
        "long_right": "->",
        "long_left": "<-",
        "long_up": "^|",
        "long_down": "v|",
        "hook_right": "->|",
        "hook_left": "|<-",
        "curve_right": ")>",
        "curve_left": "<(",
        "barbed_right": ">>|",
        "barbed_left": "|<<",
    }

    SHAPES: Mapping[str, str] = {
        "circle": "O",
        "circle_empty": "o",
        "square": "#",
        "square_empty": "[]",
        "diamond": "<>",
        "diamond_empty": "< >",
        "triangle": "^",
        "triangle_empty": "/\\",
        "circle_medium": "(O)",
        "bullseye": "(*)",
        "ellipse_h": "(--)",
        "ellipse_v": "(| |)",
        "hexagon": "[#6]",
        "hexagon_empty": "[ 6 ]",
        "pentagon": "[#5]",
        "pentagon_empty": "[ 5 ]",
        "octagon": "[#8]",
        "octagon_empty": "[ 8 ]",
        "lozenge": "<>",
        "parallelogram": "/##/",
        "parallelogram_empty": "/__/",
    }

    STATUS: Mapping[str, str] = {
        "check": "[OK]",
        "cross": "[X]",
        "warning": "[!]",
        "info": "[i]",
        "star": "*",
        "star_empty": ".",
        "heart": "<3",
        "heart_empty": "</3",
        "skull": "[X]",
        "sword": "|-",
        "shield": "[=]",
        "potion": "(~)",
        "coin": "$",
        "gem": "*",
        "question": "[?]",
        "exclamation": "!!",
        "pause": "||",
        "play": ">",
        "stop": "[]",
        "record": "(o)",
        "hourglass": "H:",
        "success": "[OK]",
        "failure": "[X]",
        "pending": "...",
        "lock": "[#]",
        "unlock": "[ ]",
        "battery": "[|||]",
        "trash": "[X_]",
        "search": "[? ]",
        "gear": "{=}",
    }

    TREE: Mapping[str, str] = {
        "branch": "|-- ",
        "branch_last": "`-- ",
        "vertical": "|   ",
        "space": "    ",
        "branch_mid": "|- ",
        "branch_heavy": "#= ",
        "branch_last_heavy": "`= ",
        "vertical_heavy": "#   ",
    }

    LINES: Mapping[str, str] = {
        "solid": "-",
        "double": "=",
        "dashed": "~",
        "dotted": ".",
        "heavy": "=",
        "decorative": "===",
        "wavy": "~~~",
        "hair": "-",
        "thick": "####",
        "thin": "--",
        "triple": "===",
    }

    BLOCK_CHART: str = "_.-~:;!*#"

    UI_ICONS: Mapping[str, str] = {
        "folder": "[DIR]",
        "file": "[FILE]",
        "arrow_right": ">",
        "checkbox_checked": "[x]",
        "checkbox_unchecked": "[ ]",
        "radio_selected": "(*)",
        "radio_unselected": "( )",
        "quote": "|",
        "checkmark": "[OK]",
        "home": "[HOME]",
        "settings": "[CFG]",
        "download": "[DL]",
        "upload": "[UP]",
        "link": "[LINK]",
        "unlink": "[X-LINK]",
        "terminal": "[TERM]",
        "keyboard": "[KB]",
        "mouse": "[MS]",
        "clipboard": "[CLIP]",
        "pin": "[PIN]",
        "bell": "[BEL]",
        "bell_off": "[bel]",
        "eye": "[EYE]",
        "eye_off": "[--EYE]",
    }

    SEPARATORS: Mapping[str, str] = {
        "arrow": ">",
        "dash": "--",
        "pipe": "|",
        "double_pipe": "||",
        "middot": ".",
        "bull": "*",
        "chevron": ">>",
        "slash": "/",
        "backslash": "\\",
        "tilde": "~",
    }


class SymbolManager:
    """Manages symbol mode selection and registry delegation."""

    def __init__(self) -> None:
        self._unicode_registry = UnicodeSymbolRegistry()
        self._ascii_registry = ASCIISymbolRegistry()
        self._ascii_mode = self._detect_ascii_mode()

    def _detect_ascii_mode(self) -> bool:
        """Detect if ASCII mode should be used based on multiple criteria."""
        # Check environment variable
        if os.environ.get("TEXTFORGE_ASCII_MODE", "").lower() in ("1", "true", "yes"):
            return True

        # Check terminal capabilities (basic detection)
        if hasattr(sys.stdout, "encoding") and sys.stdout.encoding:
            encoding = sys.stdout.encoding.lower()
            # Use ASCII for Windows consoles that don't support Unicode well
            if "cp1252" in encoding or "cp437" in encoding:
                return True
            # Use ASCII for basic ASCII terminals
            if encoding in ("ascii", "us-ascii"):
                return True

        # Check for Windows without modern terminal
        if os.name == "nt":
            # Could add more sophisticated Windows terminal detection here
            # For now, default to Unicode on Windows (modern terminals support it)
            pass

        return False

    def set_ascii_mode(self, ascii_mode: bool) -> None:
        """Explicitly set ASCII mode."""
        self._ascii_mode = ascii_mode

    def is_ascii_mode(self) -> bool:
        """Check if ASCII mode is active."""
        return self._ascii_mode

    def get_registry(self):
        """Get the appropriate symbol registry."""
        return self._ascii_registry if self._ascii_mode else self._unicode_registry

    def get_symbols(self, style: str = "box") -> dict[str, str]:
        """Get symbol preset for a style."""
        registry = self.get_registry()
        return registry.PRESETS.get(style, registry.PRESETS["box"])

    @property
    def BAR_FILLED(self) -> str:
        return self.get_registry().BAR_FILLED

    @property
    def BAR_EMPTY(self) -> str:
        return self.get_registry().BAR_EMPTY

    @property
    def BAR_PARTIAL(self) -> list[str]:
        return self.get_registry().BAR_PARTIAL

    @property
    def ARROWS(self) -> Mapping[str, str]:
        return self.get_registry().ARROWS

    @property
    def SHAPES(self) -> Mapping[str, str]:
        return self.get_registry().SHAPES

    @property
    def STATUS(self) -> Mapping[str, str]:
        return self.get_registry().STATUS

    @property
    def TREE(self) -> Mapping[str, str]:
        return self.get_registry().TREE

    @property
    def LINES(self) -> Mapping[str, str]:
        return self.get_registry().LINES

    @property
    def BLOCK_CHART(self) -> str:
        return self.get_registry().BLOCK_CHART

    @property
    def UI_ICONS(self) -> Mapping[str, str]:
        return self.get_registry().UI_ICONS

    @property
    def SEPARATORS(self) -> Mapping[str, str]:
        return self.get_registry().SEPARATORS


# Global symbol manager instance
_default_manager = SymbolManager()


class Symbols:
    """Facade for symbol management with automatic mode detection.

    Automatically detects terminal capabilities and environment settings
    to choose between Unicode and ASCII symbol sets.
    """

    # Legacy compatibility - delegate to manager
    @staticmethod
    def get_symbols(style: str = "box", ascii_mode: bool | None = None) -> dict[str, str]:
        """Return the symbol preset for a style.

        Args:
            style: Symbol preset name (box, double, rounded, etc.)
            ascii_mode: Force ASCII mode (True), Unicode mode (False), or auto-detect (None)
        """
        if ascii_mode is True:
            _default_manager.set_ascii_mode(True)
        elif ascii_mode is False:
            _default_manager.set_ascii_mode(False)
        # If None, keep current auto-detected mode

        return _default_manager.get_symbols(style)

    @staticmethod
    def BAR_FILLED() -> str:
        """Filled bar character."""
        return _default_manager.BAR_FILLED

    @staticmethod
    def BAR_EMPTY() -> str:
        """Empty bar character."""
        return _default_manager.BAR_EMPTY

    @staticmethod
    def BAR_PARTIAL() -> list[str]:
        """Partial bar characters."""
        return _default_manager.BAR_PARTIAL

    @staticmethod
    def ARROWS() -> Mapping[str, str]:
        """Arrow symbols."""
        return _default_manager.ARROWS

    @staticmethod
    def SHAPES() -> Mapping[str, str]:
        """Shape symbols."""
        return _default_manager.SHAPES

    @staticmethod
    def STATUS() -> Mapping[str, str]:
        """Status symbols."""
        return _default_manager.STATUS

    @staticmethod
    def TREE() -> Mapping[str, str]:
        """Tree drawing symbols."""
        return _default_manager.TREE

    @staticmethod
    def LINES() -> Mapping[str, str]:
        """Line drawing symbols."""
        return _default_manager.LINES

    @staticmethod
    def BLOCK_CHART() -> str:
        """Block chart characters."""
        return _default_manager.BLOCK_CHART

    @staticmethod
    def UI_ICONS() -> Mapping[str, str]:
        """UI icon symbols."""
        return _default_manager.UI_ICONS

    @staticmethod
    def SEPARATORS() -> Mapping[str, str]:
        """Separator symbols."""
        return _default_manager.SEPARATORS

    @staticmethod
    def set_ascii_mode(ascii_mode: bool) -> None:
        """Force ASCII mode globally."""
        _default_manager.set_ascii_mode(ascii_mode)

    @staticmethod
    def is_ascii_mode() -> bool:
        """Check if ASCII mode is currently active."""
        return _default_manager.is_ascii_mode()

    # Legacy compatibility properties
    PRESETS = property(lambda self: _default_manager.get_registry().PRESETS)
    ASCII_PRINTABLE = "".join(chr(c) for c in range(0x20, 0x7F))
