"""Theme registry and palette management for Textforge.

Supports per-thread palette overrides so individual consoles can apply
temporary theme-token substitutions during rendering without affecting
global state.
"""

from __future__ import annotations

import threading
from contextlib import contextmanager
from dataclasses import dataclass


@dataclass(slots=True, frozen=True)
class Theme:
    """A theme defines semantic color tokens mapped to color specs.

    Color specs can be named colors (e.g., "cyan"), hex strings (e.g., "#00ffff"),
    or raw ANSI codes. Resolution to ANSI is delegated to `Color.get_color`.
    """

    name: str
    palette: dict[str, str]


class ThemeManager:
    """A minimal theme registry with a current-theme pointer."""

    _themes: dict[str, Theme] = {}
    _current: str | None = None
    _version: int = 0  # increment to invalidate caches on theme changes
    _tls = threading.local()

    @classmethod
    def register(cls, theme: Theme) -> None:
        cls._themes[theme.name] = theme
        if cls._current is None:
            cls._current = theme.name
        cls._version += 1

    @classmethod
    def set_current(cls, theme_name: str) -> None:
        if theme_name in cls._themes:
            cls._current = theme_name
            cls._version += 1

    @classmethod
    def get_current(cls) -> Theme | None:
        if cls._current is None:
            return None
        return cls._themes.get(cls._current)

    @classmethod
    def get_palette_value(cls, token: str) -> str | None:
        theme = cls.get_current()
        if not theme:
            return None
        # Check thread-local overrides (top of stack wins)
        overrides = getattr(cls._tls, "overrides_stack", None)
        if overrides:
            for ov in reversed(overrides):
                if token in ov:
                    return ov[token]
        return theme.palette.get(token)

    @classmethod
    def available(cls) -> dict[str, Theme]:
        return dict(cls._themes)

    @classmethod
    def inherit(cls, name: str, base: str, overrides: dict[str, str]) -> None:
        """Create a new theme inheriting from an existing one with overrides."""
        parent = cls._themes.get(base)
        if not parent:
            return
        merged = dict(parent.palette)
        merged.update(overrides)
        cls.register(Theme(name=name, palette=merged))

    @classmethod
    def update_theme(cls, name: str, updates: dict[str, str]) -> None:
        """Update an existing theme's palette with new values."""
        theme = cls._themes.get(name)
        if not theme:
            return
        updated_palette = dict(theme.palette)
        updated_palette.update(updates)
        cls.register(Theme(name=name, palette=updated_palette))

    @classmethod
    def get_version(cls) -> int:
        """Return a monotonically increasing integer that changes on theme updates."""
        return cls._version

    # Per-thread override management
    @classmethod
    def push_overrides(cls, overrides: dict[str, str] | None) -> None:
        if not overrides:
            return
        stack = getattr(cls._tls, "overrides_stack", None)
        if stack is None:
            stack = []
            cls._tls.overrides_stack = stack
        stack.append(dict(overrides))
        cls._version += 1

    @classmethod
    def pop_overrides(cls) -> None:
        stack = getattr(cls._tls, "overrides_stack", None)
        if stack:
            stack.pop()
            cls._version += 1

    @classmethod
    @contextmanager
    def override(cls, overrides: dict[str, str] | None):
        try:
            cls.push_overrides(overrides)
            yield
        finally:
            cls.pop_overrides()


# Register a sensible default theme on import
ThemeManager.register(
    Theme(
        name="default",
        palette={
            "primary": "#4ea1ff",
            "secondary": "#9aa5b1",
            "accent": "#ffb020",
            "info": "#2dc6ff",
            "success": "#22c55e",
            "warning": "#f59e0b",
            "error": "#ef4444",
            "muted": "#94a3b8",
            "fg": "#e5e7eb",
            "bg": "#111827",
        },
    )
)
