"""Centralized color palettes, ANSI effect maps, and name → value tables.

This module intentionally contains only data constants and tiny helpers.
Logic that converts palette values to ANSI lives in `ansi.py` and higher-level
APIs live in `colors.py`.
"""

from __future__ import annotations

# Named foreground color → ANSI escape (mostly 4/8-bit shortcuts)
NAMED_COLORS: dict[str, str] = {
    "red": "\033[91m",
    "green": "\033[92m",
    "yellow": "\033[93m",
    "blue": "\033[94m",
    "magenta": "\033[95m",
    "cyan": "\033[96m",
    "white": "\033[97m",
    "purple": "\033[95m",
    "orange": "\033[38;5;208m",
    "brown": "\033[38;5;94m",
    "gray": "\033[90m",
    "light_gray": "\033[37m",
    "light_red": "\033[91m",
    "light_green": "\033[92m",
    "light_yellow": "\033[93m",
    "light_blue": "\033[94m",
    "light_magenta": "\033[95m",
    "light_cyan": "\033[96m",
    "light_white": "\033[97m",
    "black": "\033[30m",
    "dark_gray": "\033[90m",
    "light_purple": "\033[95m",
    "light_orange": "\033[38;5;214m",
    "light_brown": "\033[38;5;130m",
    "gold": "\033[38;5;220m",
    "pink": "\033[38;5;205m",
    "lime": "\033[38;5;46m",
    "navy": "\033[38;5;17m",
    "teal": "\033[38;5;30m",
    "silver": "\033[38;5;7m",
    "maroon": "\033[38;5;1m",
    "olive": "\033[38;5;58m",
}

# Named color → hex value (canonical palette for conversions)
NAMED_HEX: dict[str, str] = {
    "red": "#ef4444",
    "green": "#22c55e",
    "yellow": "#facc15",
    "blue": "#3b82f6",
    "magenta": "#d946ef",
    "cyan": "#06b6d4",
    "white": "#f9fafb",
    "purple": "#a855f7",
    "orange": "#fb923c",
    "brown": "#92400e",
    "gray": "#9ca3af",
    "light_gray": "#d1d5db",
    "light_red": "#f87171",
    "light_green": "#4ade80",
    "light_yellow": "#fde047",
    "light_blue": "#60a5fa",
    "light_magenta": "#f472b6",
    "light_cyan": "#67e8f9",
    "light_white": "#ffffff",
    "black": "#111827",
    "dark_gray": "#6b7280",
    "light_purple": "#c084fc",
    "light_orange": "#f97316",
    "light_brown": "#b45309",
    "gold": "#fbbf24",
    "pink": "#ec4899",
    "lime": "#84cc16",
    "navy": "#1d4ed8",
    "teal": "#14b8a6",
    "silver": "#e5e7eb",
    "maroon": "#b91c1c",
    "olive": "#4d7c0f",
}

# 4-bit/8-bit ANSI code → RGB approximations (for reverse lookups)
ANSI_4BIT_RGB: dict[str, tuple[int, int, int]] = {
    "30": (0, 0, 0),
    "31": (187, 0, 0),
    "32": (0, 187, 0),
    "33": (187, 187, 0),
    "34": (0, 0, 187),
    "35": (187, 0, 187),
    "36": (0, 187, 187),
    "37": (255, 255, 255),
    "90": (85, 85, 85),
    "91": (255, 85, 85),
    "92": (85, 255, 85),
    "93": (255, 255, 85),
    "94": (85, 85, 255),
    "95": (255, 85, 255),
    "96": (85, 255, 255),
    "97": (255, 255, 255),
}

# Background named colors
BG_COLORS: dict[str, str] = {
    "bg_red": "\033[101m",
    "bg_green": "\033[102m",
    "bg_yellow": "\033[103m",
    "bg_blue": "\033[104m",
    "bg_magenta": "\033[105m",
    "bg_cyan": "\033[106m",
    "bg_white": "\033[107m",
    "bg_black": "\033[40m",
    "bg_gray": "\033[100m",
    "bg_light_gray": "\033[47m",
    "bg_orange": "\033[48;5;208m",
    "bg_brown": "\033[48;5;94m",
    "bg_purple": "\033[48;5;93m",
    "bg_gold": "\033[48;5;220m",
    "bg_pink": "\033[48;5;205m",
    "bg_lime": "\033[48;5;46m",
    "bg_navy": "\033[48;5;17m",
    "bg_teal": "\033[48;5;30m",
}

# Text effects
EFFECTS: dict[str, str] = {
    "reset": "\033[0m",
    "bold": "\033[1m",
    "dim": "\033[2m",
    "italic": "\033[3m",
    "underline": "\033[4m",
    "double_underline": "\033[21m",
    "blink": "\033[5m",
    "reverse": "\033[7m",
    "hidden": "\033[8m",
    "strikethrough": "\033[9m",
    "overline": "\033[53m",
    "framed": "\033[51m",
    "encircled": "\033[52m",
}

RESET = EFFECTS["reset"]

__all__ = [
    "NAMED_COLORS",
    "NAMED_HEX",
    "ANSI_4BIT_RGB",
    "BG_COLORS",
    "EFFECTS",
    "RESET",
]
