"""
Color utilities for Textforge.

This module now acts as a high-level facade that builds on split modules:
- `textforge/style/palette.py` for static maps and effect codes
- `textforge/style/ansi.py` for low-level conversions
- `textforge/style/gradients.py` for gradient rendering
"""

from __future__ import annotations

import colorsys
import re

from . import ansi as _ansi
from .gradients import apply_gradient as _apply_gradient
from .palette import (
    ANSI_4BIT_RGB,
    BG_COLORS,
    EFFECTS,
    NAMED_COLORS,
    NAMED_HEX,
)
from .palette import (
    RESET as RESET_CODE,
)


class Color:
    """Manages ANSI color codes and conversions."""

    # Back-compat: expose underlying maps as class attributes
    NAMED_COLORS = NAMED_COLORS
    NAMED_HEX = NAMED_HEX
    ANSI_4BIT_RGB = ANSI_4BIT_RGB
    BG_COLORS = BG_COLORS
    EFFECTS = EFFECTS

    # Re-exported effect constants for convenience
    RESET = RESET_CODE
    BOLD = EFFECTS["bold"]
    DIM = EFFECTS["dim"]
    ITALIC = EFFECTS["italic"]
    UNDERLINE = EFFECTS["underline"]
    BLINK = EFFECTS["blink"]
    REVERSE = EFFECTS["reverse"]
    HIDDEN = EFFECTS["hidden"]
    STRIKETHROUGH = EFFECTS["strikethrough"]

    # Cache for resolved color specs keyed by (theme_version, color_spec)
    _CACHE: dict[tuple[int, str], str] = {}
    _RGB_CACHE: dict[tuple[int, str], tuple[int, int, int]] = {}
    _LAST_THEME_VERSION: int | None = None

    @staticmethod
    def get_color(color_spec: str | None) -> str:
        """
        Convert color specification to ANSI code.

        Args:
            color_spec: Color name or ANSI code.

        Returns:
            ANSI escape code for the color, or empty string if invalid.
        """
        if not color_spec:
            return ""

        color_spec = color_spec.strip()

        # Import ThemeManager lazily to avoid circular imports
        try:
            from .themes import ThemeManager
        except Exception:
            ThemeManager = None

        theme_version = ThemeManager.get_version() if ThemeManager is not None else -1

        # Invalidate cache if theme version changed (or first run)
        if Color._LAST_THEME_VERSION is None or Color._LAST_THEME_VERSION != theme_version:
            Color._CACHE.clear()
            Color._RGB_CACHE.clear()
            Color._LAST_THEME_VERSION = theme_version

        cache_key = (theme_version, color_spec)
        cached = Color._CACHE.get(cache_key)
        if cached is not None:
            return cached

        # Helper to resolve a theme token to a concrete color spec
        def resolve_theme_token(token: str) -> str:
            if ThemeManager is None:
                return ""
            value = ThemeManager.get_palette_value(token)
            return value or ""

        low = color_spec.lower()

        result: str

        if low.startswith("var(") and low.endswith(")"):
            inner = low[4:-1].strip()
            resolved = resolve_theme_token(inner)
            result = Color.get_color(resolved) if resolved else ""
        elif color_spec.startswith("$"):
            token = color_spec[1:]
            resolved = resolve_theme_token(token)
            result = Color.get_color(resolved) if resolved else ""
        elif low in NAMED_COLORS:
            result = NAMED_COLORS[low]
        elif low.startswith("#"):
            result = _ansi.hex_to_ansi(low)
        elif "," in low:
            result = _ansi.rgb_str_to_ansi(low)
        elif low in EFFECTS:
            result = EFFECTS[low]
        else:
            # Theme token as bare word
            if ThemeManager is not None:
                resolved = resolve_theme_token(low)
                result = Color.get_color(resolved) if resolved else ""
            else:
                result = ""

        Color._CACHE[cache_key] = result
        return result

    @staticmethod
    def get_bg_color(color_spec: str | None) -> str:
        """
        Convert color specification to ANSI background code, honoring theme tokens.

        Accepts the same inputs as get_color, but returns a background (48;2) code.
        """
        if not color_spec:
            return ""

        # Import ThemeManager lazily to avoid circular imports
        try:
            from .themes import ThemeManager
        except Exception:
            ThemeManager = None

        theme_version = ThemeManager.get_version() if ThemeManager is not None else -1

        # Invalidate cache if theme version changed (or first run)
        if Color._LAST_THEME_VERSION is None or Color._LAST_THEME_VERSION != theme_version:
            Color._CACHE.clear()
            Color._RGB_CACHE.clear()
            Color._LAST_THEME_VERSION = theme_version

        cache_key = (theme_version, f"bg::{color_spec}")
        cached = Color._CACHE.get(cache_key)
        if cached is not None:
            return cached

        low = color_spec.strip().lower()

        def resolve_theme_token(token: str) -> str:
            if ThemeManager is None:
                return ""
            value = ThemeManager.get_palette_value(token)
            return value or ""

        if low.startswith("var(") and low.endswith(")"):
            inner = low[4:-1].strip()
            resolved = resolve_theme_token(inner)
            result = Color.get_bg_color(resolved) if resolved else ""
        elif color_spec.startswith("$"):
            token = color_spec[1:]
            resolved = resolve_theme_token(token)
            result = Color.get_bg_color(resolved) if resolved else ""
        elif low.startswith("bg#"):
            result = _ansi.hex_to_bg_ansi(low[2:])
        elif low.startswith("bg:"):
            result = _ansi.rgb_str_to_bg_ansi(low[3:])
        elif low in BG_COLORS:
            result = BG_COLORS[low]
        elif low.startswith("#"):
            result = _ansi.hex_to_bg_ansi(low)
        elif "," in low:
            result = _ansi.rgb_str_to_bg_ansi(low)
        else:
            # Try themed background: treat as token first; then fallback by converting fg to bg
            resolved = resolve_theme_token(low)
            if resolved:
                result = Color.get_bg_color(resolved)
            else:
                fg = Color.get_color(low)
                result = fg.replace("[38;", "[48;") if "[38;" in fg else ""

        Color._CACHE[cache_key] = result
        return result

    @staticmethod
    def hex_to_rgb(hex_color: str) -> tuple[int, int, int] | None:
        """Convert #RRGGBB string to an RGB tuple."""
        return _ansi.hex_to_rgb(hex_color)

    @staticmethod
    def rgb_to_hex(rgb: tuple[int, int, int]) -> str:
        """Return a #RRGGBB string for the given RGB tuple."""
        r, g, b = (max(0, min(255, int(v))) for v in rgb)
        return f"#{r:02x}{g:02x}{b:02x}"

    @staticmethod
    def hex_to_ansi(hex_color: str) -> str:
        """Convert #RRGGBB to ANSI 24-bit color."""
        return _ansi.hex_to_ansi(hex_color)

    @staticmethod
    def resolve_rgb(spec: str | None) -> tuple[int, int, int] | None:
        """Resolve a color specification into an RGB tuple if possible.

        Accepts theme tokens, named colors, hex strings, and comma-delimited
        RGB strings. Returns None when the spec cannot be resolved.
        """
        if not spec:
            return None

        s = spec.strip().lower()
        # Theme manager indirection
        try:
            from .themes import ThemeManager
        except Exception:
            ThemeManager = None

        if s.startswith("var(") and s.endswith(")"):
            inner = s[4:-1].strip()
            if ThemeManager is not None:
                resolved = ThemeManager.get_palette_value(inner)
                return Color.resolve_rgb(resolved)

        if s.startswith("$") and ThemeManager is not None:
            resolved = ThemeManager.get_palette_value(s[1:])
            return Color.resolve_rgb(resolved)

        if s in NAMED_HEX:
            return _ansi.hex_to_rgb(NAMED_HEX[s])

        if s.startswith("#"):
            return Color.hex_to_rgb(s)

        if "," in s:
            parts = s.split(",")
            if len(parts) == 3:
                try:
                    r, g, b = (int(p.strip()) for p in parts)
                except ValueError:
                    return None
                return (max(0, min(255, r)), max(0, min(255, g)), max(0, min(255, b)))

        # Attempt parse from 8/4-bit ANSI names via known map
        if s in NAMED_COLORS:
            code = NAMED_COLORS[s]
            # Extract number like 91 from "\033[91m"
            import re as _re
            m = _re.search(r"\[(\d+)m", code)
            if m:
                rgb = ANSI_4BIT_RGB.get(m.group(1))
                if rgb:
                    return rgb
        return None

    @staticmethod
    def blend(rgb_a: tuple[int, int, int], rgb_b: tuple[int, int, int], t: float) -> tuple[int, int, int]:
        """Linear blend between two RGB tuples."""
        t = max(0.0, min(1.0, t))
        return (
            int(rgb_a[0] + (rgb_b[0] - rgb_a[0]) * t),
            int(rgb_a[1] + (rgb_b[1] - rgb_a[1]) * t),
            int(rgb_a[2] + (rgb_b[2] - rgb_a[2]) * t),
        )

    @staticmethod
    def rgb_to_hsv(r: int, g: int, b: int) -> tuple[float, float, float]:
        """Convert RGB to HSV where H is degrees 0-360."""
        h, s, v = colorsys.rgb_to_hsv(r / 255.0, g / 255.0, b / 255.0)
        return (h * 360.0, s, v)

    @staticmethod
    def hsv_to_rgb(h: float, s: float, v: float) -> tuple[int, int, int]:
        """Convert HSV (degrees, 0-1, 0-1) to RGB tuple."""
        r, g, b = colorsys.hsv_to_rgb((h % 360) / 360.0, max(0.0, min(1.0, s)), max(0.0, min(1.0, v)))
        return (int(round(r * 255)), int(round(g * 255)), int(round(b * 255)))

    @staticmethod
    def rgb_to_hsl(r: int, g: int, b: int) -> tuple[float, float, float]:
        """Convert RGB to HSL where H is degrees."""
        h, light, s = colorsys.rgb_to_hls(r / 255.0, g / 255.0, b / 255.0)
        return (h * 360.0, s, light)

    @staticmethod
    def hsl_to_rgb(h: float, s: float, light: float) -> tuple[int, int, int]:
        """Convert HSL (degrees, sat, light) to RGB tuple."""
        r, g, b = colorsys.hls_to_rgb((h % 360) / 360.0, max(0.0, min(1.0, light)), max(0.0, min(1.0, s)))
        return (int(round(r * 255)), int(round(g * 255)), int(round(b * 255)))

    @staticmethod
    def adjust_luminance(rgb: tuple[int, int, int], amount: float) -> tuple[int, int, int]:
        """Lighten (>0) or darken (<0) an RGB color by adjusting HSL lightness."""
        h, s, light = Color.rgb_to_hsl(*rgb)
        light = max(0.0, min(1.0, light + amount))
        return Color.hsl_to_rgb(h, s, light)

    @staticmethod
    def mix(spec_a: str, spec_b: str, t: float) -> str:
        """Blend two color specs and return the resulting ANSI code."""
        rgb_a = Color.resolve_rgb(spec_a)
        rgb_b = Color.resolve_rgb(spec_b)
        if rgb_a is None or rgb_b is None:
            return Color.get_color(spec_a if t < 0.5 else spec_b)
        rgb = Color.blend(rgb_a, rgb_b, t)
        return Color.rgb_to_ansi(rgb)

    @staticmethod
    def rgb_to_ansi(rgb: tuple[int, int, int]) -> str:
        return _ansi.rgb_to_ansi(rgb)

    @staticmethod
    def rgb_str_to_ansi(rgb_color: str) -> str:
        """Convert 'r,g,b' to ANSI 24-bit color."""
        return _ansi.rgb_str_to_ansi(rgb_color)

    @staticmethod
    def hex_to_bg_ansi(hex_color: str) -> str:
        """Convert #RRGGBB to ANSI 24-bit background color."""
        return _ansi.hex_to_bg_ansi(hex_color)

    @staticmethod
    def rgb_to_bg_ansi(rgb_color: str) -> str:
        """Convert 'r,g,b' to ANSI 24-bit background color."""
        return _ansi.rgb_str_to_bg_ansi(rgb_color)

    @staticmethod
    def apply_inline_markup(text: str) -> str:
        """
        Process inline color markup in text.

        Usage: "This is [red]red text[reset] and [blue]blue text[reset]"
        Supports: colors, bg_colors, bold, italic, underline, etc.

        Args:
            text: Text with inline markup tags

        Returns:
            Text with ANSI codes applied
        """

        # Protect escaped brackets: "[[" "]]" and backslash-escaped \[ \]
        LBR = "\x00__LBRACKET__\x00"
        RBR = "\x00__RBRACKET__\x00"
        text = text.replace("[[", LBR).replace("]]", RBR)
        text = text.replace("\\[", LBR).replace("\\]", RBR)

        def replace_tag(match: re.Match) -> str:
            tag_raw = match.group(1).strip()
            tag = tag_raw.lower()

            # Lazily import ThemeManager
            try:
                from .themes import ThemeManager
            except Exception:
                ThemeManager = None

            # Check effects first
            if tag in EFFECTS:
                return EFFECTS[tag]

            # Check background colors
            if tag in BG_COLORS:
                return BG_COLORS[tag]

            # Background hex (bg#RRGGBB) and RGB (bg:r,g,b)
            if tag.startswith("bg#"):
                return _ansi.hex_to_bg_ansi(tag[2:])
            if tag.startswith("bg:"):
                return _ansi.rgb_str_to_bg_ansi(tag[3:])

            # Explicit foreground/background theme tokens: fg=token, bg=token
            if tag.startswith("fg=") or tag.startswith("fg:"):
                token = tag.split("=", 1)[-1] if "=" in tag else tag.split(":", 1)[-1]
                if ThemeManager is not None:
                    spec = ThemeManager.get_palette_value(token)
                    if spec:
                        return Color.get_color(spec)
                return Color.get_color(token)

            if tag.startswith("bg=") or tag.startswith("bg:"):
                token = tag.split("=", 1)[-1] if "=" in tag else tag.split(":", 1)[-1]
                if ThemeManager is not None:
                    spec = ThemeManager.get_palette_value(token)
                else:
                    spec = None
                if spec:
                    if spec.startswith("#"):
                        return _ansi.hex_to_bg_ansi(spec)
                    if "," in spec:
                        return _ansi.rgb_str_to_bg_ansi(spec)
                    # Try named bg
                    named_bg = BG_COLORS.get(f"bg_{spec}", "")
                    if named_bg:
                        return named_bg
                    # As foreground code fallback convert 38->48
                    fg = Color.get_color(spec)
                    return fg.replace("[38;", "[48;") if "[38;" in fg else ""
                # Fallback to token directly
                if token.startswith("#"):
                    return _ansi.hex_to_bg_ansi(token)
                if "," in token:
                    return _ansi.rgb_str_to_bg_ansi(token)
                named_bg = BG_COLORS.get(f"bg_{token}", "")
                if named_bg:
                    return named_bg
                fg = Color.get_color(token)
                return fg.replace("[38;", "[48;") if "[38;" in fg else ""

            # Check colors
            if tag in NAMED_COLORS:
                return NAMED_COLORS[tag]

            # Handle hex colors
            if tag.startswith("#"):
                return _ansi.hex_to_ansi(tag)

            # Handle RGB colors
            if "," in tag:
                return _ansi.rgb_str_to_ansi(tag)

            # Theme palette bare token
            try:
                if ThemeManager is not None:
                    spec = ThemeManager.get_palette_value(tag)
                    if spec:
                        return Color.get_color(spec)
            except Exception:
                pass

            # Unknown tag - return as-is
            return match.group(0)

        rendered = re.sub(r"\[([^\]]+)\]", replace_tag, text)
        # Restore escaped braces
        rendered = rendered.replace(LBR, "[").replace(RBR, "]")
        return rendered

    @staticmethod
    def gradient(text: str, start_color: tuple[int, int, int], end_color: tuple[int, int, int]) -> str:
        rendered = _apply_gradient(text, start_color, end_color)
        return rendered + Color.RESET

    # --- Contrast and luminance helpers (moved from utils/accessibility) ---

    @staticmethod
    def luminance_from_rgb(r: int, g: int, b: int) -> float:
        srgb = [v / 255 for v in (r, g, b)]
        lin = [((c + 0.055) / 1.055) ** 2.4 if c > 0.03928 else c / 12.92 for c in srgb]
        return 0.2126 * lin[0] + 0.7152 * lin[1] + 0.0722 * lin[2]

    @staticmethod
    def contrast_ratio_rgb(rgb_a: tuple[int, int, int], rgb_b: tuple[int, int, int]) -> float:
        l1 = Color.luminance_from_rgb(*rgb_a)
        l2 = Color.luminance_from_rgb(*rgb_b)
        lighter, darker = (l1, l2) if l1 >= l2 else (l2, l1)
        return (lighter + 0.05) / (darker + 0.05)

    @staticmethod
    def is_contrast_sufficient(
        fg_spec: str,
        bg_spec: str = "#000000",
        *,
        large_text: bool = False,
    ) -> bool:
        """Return True if contrast between two color specs meets WCAG AA.

        Accepts theme tokens, named colors, hex strings, or "r,g,b" strings.
        """
        fg_rgb = Color.resolve_rgb(fg_spec)
        bg_rgb = Color.resolve_rgb(bg_spec)
        if fg_rgb is None or bg_rgb is None:
            return True
        ratio = Color.contrast_ratio_rgb(fg_rgb, bg_rgb)
        threshold = 3.0 if large_text else 4.5
        return ratio >= threshold

    @staticmethod
    def ensure_min_contrast(
        fg_spec: str,
        bg_spec: str,
        *,
        min_ratio: float = 4.5,
        max_steps: int = 40,
    ) -> str:
        """Adjust a foreground color until the requested contrast ratio is met.

        Returns a color spec as hex string when adjustment was successful; falls
        back to the original spec when inputs cannot be resolved.
        """
        fg_rgb = Color.resolve_rgb(fg_spec)
        bg_rgb = Color.resolve_rgb(bg_spec)
        if fg_rgb is None or bg_rgb is None:
            return fg_spec

        current = fg_rgb
        attempt = 0
        while Color.contrast_ratio_rgb(current, bg_rgb) < min_ratio and attempt < max_steps:
            lf = Color.luminance_from_rgb(*current)
            lb = Color.luminance_from_rgb(*bg_rgb)
            # Darken if foreground is lighter than background; otherwise lighten
            delta = -0.04 if lf > lb else 0.04
            current = Color.adjust_luminance(current, delta)
            attempt += 1
        return Color.rgb_to_hex(current)

    @staticmethod
    def simulate_color_blind(rgb: tuple[int, int, int], mode: str = "protanopia") -> tuple[int, int, int]:
        """Simulate color vision deficiencies using matrix transforms."""
        r, g, b = [c / 255.0 for c in rgb]
        matrices: dict[str, tuple[tuple[float, float, float], ...]] = {
            "protanopia": (
                (0.56667, 0.43333, 0.00000),
                (0.55833, 0.44167, 0.00000),
                (0.00000, 0.24167, 0.75833),
            ),
            "deuteranopia": (
                (0.62500, 0.37500, 0.00000),
                (0.70000, 0.30000, 0.00000),
                (0.00000, 0.30000, 0.70000),
            ),
            "tritanopia": (
                (0.95000, 0.05000, 0.00000),
                (0.00000, 0.43333, 0.56667),
                (0.00000, 0.47500, 0.52500),
            ),
        }
        mat = matrices.get(mode, matrices["protanopia"])
        r2 = r * mat[0][0] + g * mat[0][1] + b * mat[0][2]
        g2 = r * mat[1][0] + g * mat[1][1] + b * mat[1][2]
        b2 = r * mat[2][0] + g * mat[2][1] + b * mat[2][2]
        return (
            int(max(0, min(1, r2)) * 255),
            int(max(0, min(1, g2)) * 255),
            int(max(0, min(1, b2)) * 255),
        )


__all__ = ["Color"]
