"""ANSI conversion helpers.

Thin utilities for turning color specifications into ANSI escape codes.
Heavyweight caching/indirection is handled by higher-level APIs.
"""

from __future__ import annotations

from .palette import BG_COLORS, EFFECTS, NAMED_COLORS


def rgb_to_ansi(rgb: tuple[int, int, int]) -> str:
    r, g, b = (max(0, min(255, int(v))) for v in rgb)
    return f"\033[38;2;{r};{g};{b}m"


def rgb_to_bg_ansi(rgb: tuple[int, int, int]) -> str:
    r, g, b = (max(0, min(255, int(v))) for v in rgb)
    return f"\033[48;2;{r};{g};{b}m"


def hex_to_rgb(hex_color: str) -> tuple[int, int, int] | None:
    s = hex_color.strip().lstrip("#")
    if len(s) != 6:
        return None
    try:
        r = int(s[0:2], 16)
        g = int(s[2:4], 16)
        b = int(s[4:6], 16)
    except ValueError:
        return None
    return (r, g, b)


def hex_to_ansi(hex_color: str) -> str:
    rgb = hex_to_rgb(hex_color)
    return rgb_to_ansi(rgb) if rgb else ""


def hex_to_bg_ansi(hex_color: str) -> str:
    rgb = hex_to_rgb(hex_color)
    return rgb_to_bg_ansi(rgb) if rgb else ""


def parse_rgb_string(rgb_color: str) -> tuple[int, int, int] | None:
    parts = rgb_color.split(",")
    if len(parts) != 3:
        return None
    try:
        r, g, b = (int(part.strip()) for part in parts)
    except ValueError:
        return None
    return (max(0, min(255, r)), max(0, min(255, g)), max(0, min(255, b)))


def rgb_str_to_ansi(rgb_color: str) -> str:
    rgb = parse_rgb_string(rgb_color)
    return rgb_to_ansi(rgb) if rgb else ""


def rgb_str_to_bg_ansi(rgb_color: str) -> str:
    rgb = parse_rgb_string(rgb_color)
    return rgb_to_bg_ansi(rgb) if rgb else ""


def effect(code: str) -> str:
    return EFFECTS.get(code, "")


def named_fg(name: str) -> str:
    return NAMED_COLORS.get(name, "")


def named_bg(name: str) -> str:
    return BG_COLORS.get(name, "")


__all__ = [
    "rgb_to_ansi",
    "rgb_to_bg_ansi",
    "hex_to_rgb",
    "hex_to_ansi",
    "hex_to_bg_ansi",
    "parse_rgb_string",
    "rgb_str_to_ansi",
    "rgb_str_to_bg_ansi",
    "effect",
    "named_fg",
    "named_bg",
]
