"""Renderable base classes to standardize render/measure across components.

Components should derive from `RenderableBase` or mix in `MeasureMixin` to
provide consistent rendering and measurement hooks, rather than ad-hoc print
helpers. This enables composition and better testing.
"""

from __future__ import annotations

from typing import TYPE_CHECKING, Protocol

from ..core import Console, Measure, Renderable

if TYPE_CHECKING:
    from collections.abc import Iterable


class RenderableBase(Renderable):
    """Base class for components that render to strings via Console."""

    def render(self, console: Console) -> str | Iterable[str] | None:
        raise NotImplementedError

    def measure(self, console: Console) -> Measure | None:
        rendered = self.render(console)
        if rendered is None:
            return Measure(width=0, height=0)
        if isinstance(rendered, str):
            return Measure.from_text(rendered)
        return Measure.from_text("".join(rendered))


class MeasureMixin:
    """Provide a default measure() implementation using render()."""

    def measure(self: HasRender, console: Console) -> Measure | None:
        rendered = self.render(console)
        if rendered is None:
            return Measure(width=0, height=0)
        if isinstance(rendered, str):
            return Measure.from_text(rendered)
        return Measure.from_text("".join(rendered))


class HasRender(Protocol):
    def render(self, console: Console) -> str | Iterable[str] | None:
        ...
