"""Minimal Markdown to Textforge component renderer.

Supports a pragmatic subset: headings (#..####), paragraphs, bullet lists (-, *),
numbered lists (1.), fenced code blocks (```), and simple tables (pipe-separated).
"""

from __future__ import annotations

from contextlib import redirect_stdout
from io import StringIO

from ..components import ListElement, Paragraph, Table, Title


def render_markdown(md: str, width: int = 80, return_ansi: bool = False) -> str | None:
    buffer = StringIO() if return_ansi else None
    lines = md.splitlines()
    i = 0
    in_code = False
    code_lang = None
    code_lines: list[str] = []
    list_items: list[str] = []
    list_mode: str | None = None

    def flush_list() -> None:
        nonlocal list_items, list_mode
        if list_items:
            list_type = "number" if list_mode == "ol" else "bullet"
            if buffer:
                with redirect_stdout(buffer):
                    ListElement.render(list_items, width=width, list_type=list_type)
            else:
                ListElement.render(list_items, width=width, list_type=list_type)
            list_items = []
            list_mode = None

    while i < len(lines):
        line = lines[i]
        if line.strip().startswith("```"):
            if not in_code:
                in_code = True
                code_lang = line.strip().strip("`") or None
                code_lines = []
            else:
                # end code
                from ..components.data import CodeBlock

                if buffer:
                    with redirect_stdout(buffer):
                        CodeBlock.render("\n".join(code_lines), language=code_lang or None, width=width)
                else:
                    CodeBlock.render("\n".join(code_lines), language=code_lang or None, width=width)
                in_code = False
                code_lang = None
                code_lines = []
            i += 1
            continue

        if in_code:
            code_lines.append(line)
            i += 1
            continue

        if line.startswith("#"):
            flush_list()
            level = len(line) - len(line.lstrip("#"))
            text = line[level:].strip()
            if buffer:
                with redirect_stdout(buffer):
                    Title.render(text, width=width, symbol_style="rounded", color="primary" if level == 1 else None)
            else:
                Title.render(text, width=width, symbol_style="rounded", color="primary" if level == 1 else None)
            i += 1
            continue

        if line.strip().startswith(("- ", "* ")):
            if list_mode not in (None, "ul"):
                flush_list()
            list_mode = "ul"
            list_items.append(line.strip()[2:])
            i += 1
            continue

        if line.strip().startswith(tuple(f"{n}." for n in range(1, 10))):
            if list_mode not in (None, "ol"):
                flush_list()
            list_mode = "ol"
            # Split at dot
            content = line.strip().split(".", 1)[-1].strip()
            list_items.append(content)
            i += 1
            continue

        # Simple table detection: lines with pipes and a separator row
        if "|" in line and i + 1 < len(lines) and set(lines[i + 1].replace(" ", "")) <= {"|", "-", ":"}:
            flush_list()
            header = [col.strip() for col in line.strip().strip("|").split("|")]
            i += 2  # skip separator
            rows: list[list[str | int | float]] = []
            while i < len(lines) and "|" in lines[i]:
                parsed: list[str | int | float] = []
                for col in lines[i].strip().strip("|").split("|"):
                    col_str = col.strip()
                    # Try to coerce to number
                    try:
                        if "." in col_str:
                            parsed.append(float(col_str))
                        else:
                            parsed.append(int(col_str))
                    except ValueError:
                        parsed.append(col_str)
                row = parsed
                rows.append(row)
                i += 1
            if buffer:
                with redirect_stdout(buffer):
                    Table.render(header, rows, width=width)
            else:
                Table.render(header, rows, width=width)
            continue

        # Paragraphs
        if line.strip():
            flush_list()
            # Collect until blank line
            paras: list[str] = [line]
            i += 1
            while i < len(lines) and lines[i].strip():
                paras.append(lines[i])
                i += 1
            if buffer:
                with redirect_stdout(buffer):
                    Paragraph.render(" ".join(paras), width=width)
            else:
                Paragraph.render(" ".join(paras), width=width)
            continue

        else:
            flush_list()
            i += 1

    flush_list()

    if return_ansi and buffer:
        return buffer.getvalue()
    return None
