"""Markup parsing facilities.

Upgraded to include a small tokenizer that can handle nested tags, escaping,
and custom tag callbacks of the form {tag:payload}. Malformed tags degrade to
literal text. The public surface remains MarkupEngine.register_tag/render.
"""

from __future__ import annotations

import re as _re
from typing import TYPE_CHECKING

from ..style.colors import Color

if TYPE_CHECKING:
    from collections.abc import Callable


class MarkupEngine:
    """Inline markup renderer with simple custom-tag registry.

    Supports registering custom handlers for tags: {tagname:payload} → handler(payload).
    If a handler returns a string, it is injected in-place; otherwise the original tag
    is left intact. Handlers are evaluated before the stock color/effect processor.
    """

    # Global registry so CLIs and consoles can share known tags
    _global_handlers: dict[str, Callable[[str], str | None]] = {}

    def __init__(self, *, safe_mode: bool = False) -> None:
        # Start with a copy of global handlers to make instances independent
        self._tag_handlers: dict[str, Callable[[str], str | None]] = dict(self._global_handlers)
        # When enabled, all markup sequences are treated as literal text
        # and escaped so they render safely without evaluation.
        self._safe_mode = bool(safe_mode)

    def register_tag(self, name: str, handler: Callable[[str], str | None]) -> None:
        """Register a custom tag handler for this markup engine instance.

        Args:
            name: The tag name (case-insensitive).
            handler: A callable that takes a payload string and returns a replacement string or None.

        Returns:
            None
        """
        self._tag_handlers[name] = handler

    # Class-level registration for global availability
    @classmethod
    def register_global_tag(cls, name: str, handler: Callable[[str], str | None]) -> None:
        """Register a custom tag handler globally for all markup engine instances.

        Args:
            name: The tag name (case-insensitive).
            handler: A callable that takes a payload string and returns a replacement string or None.

        Returns:
            None
        """
        cls._global_handlers[name] = handler

    @classmethod
    def list_global_tags(cls) -> list[str]:
        """List all globally registered custom tag names.

        Returns:
            A sorted list of registered tag names.
        """
        return sorted(cls._global_handlers.keys())

    def render(self, text: str) -> str:
        """Render markup text with custom tags and color/effect processing.

        Processes the input text through several phases: escape handling, custom tag replacement,
        color/effect markup application, and escape restoration. In safe mode, all markup is
        escaped to render literally.

        Args:
            text: The input text containing markup.

        Returns:
            The rendered text with markup processed.
        """
        # In safe mode, neutralize markup and custom-tag syntax entirely
        if self._safe_mode:
            # Escape bracket/brace markup so it renders literally
            return (
                text.replace("[", "[[").replace("]", "]]")
                .replace("{", "{{").replace("}", "}}")
            )
        # Tokenize to protect escaped delimiters and nested constructs
        # 1) Protect escaped brackets and braces
        LBR = "\x00__LBR__\x00"
        RBR = "\x00__RBR__\x00"
        LCB = "\x00__LCB__\x00"
        RCB = "\x00__RCB__\x00"
        s = (
            text.replace("[[", LBR)
            .replace("]]", RBR)
            .replace("\\[", LBR)
            .replace("\\]", RBR)
            .replace("{{", LCB)
            .replace("}}", RCB)
            .replace("\\{", LCB)
            .replace("\\}", RCB)
        )

        # 2) Custom tags: {name:payload}
        if self._tag_handlers:
            def _custom_replace(m: _re.Match[str]) -> str:
                inner = m.group(1)
                if ":" not in inner:
                    return m.group(0)
                name, payload = inner.split(":", 1)
                name = name.strip().lower()
                payload = payload.strip()
                handler = self._tag_handlers.get(name)
                if not handler:
                    return m.group(0)
                try:
                    result = handler(payload)
                except Exception:
                    return m.group(0)
                return result if isinstance(result, str) else m.group(0)

            s = _re.sub(r"\{([^\}]+)\}", _custom_replace, s)

            # Support bracket-syntax custom tags: [name:payload]
            # Only intercept when name is a registered custom tag to avoid
            # interfering with stock color/effect tags like [red]
            def _bracket_replace(m: _re.Match[str]) -> str:
                inner = m.group(1)
                # Only treat as custom tag if it contains a ':' and the name is registered
                if ":" not in inner:
                    return m.group(0)
                name, payload = inner.split(":", 1)
                name = name.strip().lower()
                payload = payload.strip()
                handler = self._tag_handlers.get(name)
                if not handler:
                    return m.group(0)
                try:
                    result = handler(payload)
                except Exception:
                    return m.group(0)
                return result if isinstance(result, str) else m.group(0)

            # Only transform bracket custom tags that match registered names to ensure
            # idempotence when re-rendering text that already contains stock tags.
            s = _re.sub(r"\[([^\]]+)\]", _bracket_replace, s)

        # 3) Apply stock color/effect processor exactly once. If the string
        # already contains SGR sequences from a prior render, skip to ensure
        # idempotence.
        if "\x1b[" not in s:
            s = Color.apply_inline_markup(s)

        # 4) Restore escapes
        s = (
            s.replace(LBR, "[")
            .replace(RBR, "]")
            .replace(LCB, "{")
            .replace(RCB, "}")
        )
        return s
