"""ANSI/markup to SVG exporter (hardened).

Creates a monospaced text SVG from ANSI-coloured input. Handles nested SGR
sequences, resets, 24-bit colours, and background colours by drawing per-span
rectangles sized using a simple monospaced width estimate.
"""

from __future__ import annotations

import html
import re
from typing import Any


def ansi_to_svg(text: str, font_family: str = "Fira Code, monospace", font_size: int = 14, line_height: int = 18) -> str:
    # Split lines to establish canvas size
    lines = text.splitlines() or [""]
    cols = max(len(_strip_ansi(line)) for line in lines) or 1
    rows = len(lines)
    cw = font_size * 0.6  # approximate char width for monospace
    width = cols * cw + 20
    height = rows * line_height + 20

    # Begin SVG
    svg_parts: list[str] = [
        f'<svg xmlns="http://www.w3.org/2000/svg" width="{int(width)}" height="{int(height)}" viewBox="0 0 {int(width)} {int(height)}">',
        '<rect width="100%" height="100%" fill="#111827"/>',
        f'<g font-family="{html.escape(font_family)}" font-size="{font_size}" fill="#e5e7eb">',
    ]

    y = 10 + font_size
    for raw_line in lines:
        segments = _parse_ansi_segments(raw_line)
        # Draw background rectangles per segment first
        x = 10.0
        for text_seg, style in segments:
            if not text_seg:
                continue
            seg_len = len(text_seg)
            seg_w = seg_len * cw
            bg = style.get("bg")
            if bg:
                # Position rectangle roughly behind the text baseline
                y_rect = y - font_size * 0.8
                svg_parts.append(
                    f'<rect x="{x:.1f}" y="{y_rect:.1f}" width="{seg_w:.1f}" height="{line_height:.1f}" fill="{bg}"/>'
                )
            x += seg_w

        # Now draw the text with tspans per segment for foreground colour
        svg_parts.append(f'<text x="10" y="{y}">')
        for text_seg, style in segments:
            if not text_seg:
                continue
            fg = style.get("fg") or "#e5e7eb"
            # Escape text and attribute content to prevent injection
            safe_fg = html.escape(fg, quote=True)
            safe_text = html.escape(text_seg)
            svg_parts.append(f'<tspan style="fill:{safe_fg}">{safe_text}</tspan>')
        svg_parts.append("</text>")

        y += line_height

    svg_parts.append("</g></svg>")
    return "".join(svg_parts)


_SGR_RE = re.compile(r"\x1b\[([0-9;]*)m")

_FG_MAP: dict[int, str] = {
    30: "#000000",
    31: "#ef4444",
    32: "#22c55e",
    33: "#f59e0b",
    34: "#3b82f6",
    35: "#a855f7",
    36: "#06b6d4",
    37: "#e5e7eb",
    90: "#6b7280",
    91: "#f87171",
    92: "#34d399",
    93: "#fde047",
    94: "#60a5fa",
    95: "#c084fc",
    96: "#22d3ee",
    97: "#f3f4f6",
}

_BG_MAP: dict[int, str] = {
    40: "#000000",
    41: "#ef4444",
    42: "#22c55e",
    43: "#f59e0b",
    44: "#3b82f6",
    45: "#a855f7",
    46: "#06b6d4",
    47: "#e5e7eb",
    100: "#6b7280",
    101: "#f87171",
    102: "#34d399",
    103: "#fde047",
    104: "#60a5fa",
    105: "#c084fc",
    106: "#22d3ee",
    107: "#f3f4f6",
}


def _parse_ansi_segments(line: str) -> list[tuple[str, dict[str, Any]]]:
    """Return list of (text, style) segments for a single line.

    Style keys: fg (CSS color), bg (CSS color). Other stylistic SGRs are ignored
    for SVG output as they are not directly supported in text rendering.
    """
    segments: list[tuple[str, dict[str, Any]]] = []
    state: dict[str, Any] = {}
    buf: list[str] = []
    pos = 0
    for m in _SGR_RE.finditer(line):
        # Flush preceding text
        if m.start() > pos:
            buf.append(line[pos:m.start()])
        # Commit buffer into a segment with current style
        if buf:
            segments.append(("".join(buf), dict(state)))
            buf.clear()
        params = m.group(1)
        if params == "" or params == "0":
            state.clear()
        else:
            parts = params.split(";")
            i = 0
            while i < len(parts):
                p = parts[i]
                if not p:
                    i += 1
                    continue
                code = int(p)
                if code == 0:
                    state.clear()
                elif code in _FG_MAP:
                    state["fg"] = _FG_MAP[code]
                elif code in _BG_MAP:
                    state["bg"] = _BG_MAP[code]
                elif code in (38, 48) and i + 4 < len(parts) and parts[i + 1] == "2":
                    try:
                        r = int(parts[i + 2])
                        g = int(parts[i + 3])
                        b = int(parts[i + 4])
                        css = f"rgb({r},{g},{b})"
                        if code == 38:
                            state["fg"] = css
                        else:
                            state["bg"] = css
                    except Exception:
                        pass
                    i += 4
                elif code == 39:
                    state.pop("fg", None)
                elif code == 49:
                    state.pop("bg", None)
                i += 1
        pos = m.end()
    if pos < len(line):
        buf.append(line[pos:])
    if buf:
        segments.append(("".join(buf), dict(state)))
    return segments


def _strip_ansi(text: str) -> str:
    return re.sub(r"\x1b\[[0-9;]*m", "", text)
