"""Render-tree exporters that turn any renderable into ANSI/HTML/SVG."""

from __future__ import annotations

from io import StringIO
from typing import Any

from ..core import Console
from .html import ansi_to_html
from .svg import ansi_to_svg


def render_to_ansi(obj: Any, *, markup: bool = True, safe_mode: bool = False) -> str:  # Any: accepts any renderable-like object
    """Render any Textforge renderable-like object to an ANSI string."""
    from ..markup.engine import MarkupEngine

    buffer = StringIO()
    # Propagate safe mode into the markup engine if markup processing is enabled
    console = Console(buffer, markup=MarkupEngine(safe_mode=safe_mode))
    if isinstance(obj, str) and markup:
        obj = console.markup_engine.render(obj)
    console.print(obj, end="", markup=False)
    return buffer.getvalue()


def render_to_html(obj: Any, *, markup: bool = True, safe_mode: bool = False) -> str:  # Any: accepts any renderable-like object
    return ansi_to_html(render_to_ansi(obj, markup=markup, safe_mode=safe_mode))


def render_to_svg(obj: Any, *, markup: bool = True, safe_mode: bool = False) -> str:  # Any: accepts any renderable-like object
    return ansi_to_svg(render_to_ansi(obj, markup=markup, safe_mode=safe_mode))
