"""Minimal PDF exporter for ANSI text (hardened; supports multi-page).

Generates a PDF (Letter size by default) with monospaced text. This is a
pure-Python generator that writes a valid PDF without external dependencies.
Currently strips ANSI styling, but supports multiple pages when content
exceeds the available vertical space.
"""

from __future__ import annotations

import re

_ANSI_RE = re.compile(r"\x1b\[[0-9;]*m")


class _PDFObjectWriter:
    """Manages sequential allocation of PDF object IDs."""

    def __init__(self) -> None:
        self._next_id = 1
        self._objects: list[bytes] = []
        self._xref_positions: list[int] = []

    def add_object(self, content: str) -> int:
        """Add a PDF object and return its ID."""
        obj_id = self._next_id
        self._next_id += 1

        # Record position for xref table
        self._xref_positions.append(sum(len(obj) for obj in self._objects))

        # Add the object
        self._objects.append(content.encode("latin-1"))

        return obj_id

    def get_objects(self) -> list[bytes]:
        """Get all objects."""
        return self._objects

    def get_xref_positions(self) -> list[int]:
        """Get xref positions."""
        return self._xref_positions


def _strip_ansi(text: str) -> str:
    return _ANSI_RE.sub("", text)


def _escape_pdf_text(s: str) -> str:
    # Escape characters significant to PDF text operators and replace
    # unprintable characters to guard against PDF injection.
    safe = s.replace("\\", r"\\").replace("(", r"\(").replace(")", r"\)")
    # Replace control characters with visible placeholders
    safe = "".join(ch if (32 <= ord(ch) <= 126 or ch in "\t\n\r") else "?" for ch in safe)
    return safe


def ansi_to_pdf(
    text: str,
    *,
    font_size: int = 12,
    line_height: int | None = None,
    page_width: int = 612,
    page_height: int = 792,
    top_margin: int = 36,
    left_margin: int = 36,
) -> bytes:
    """Return a bytestring for a minimal PDF rendering the given text.

    - page size defaults to Letter (612x792 points)
    - renders with built-in Courier font
    - creates additional pages when content overflows the first page
    """
    plain = _strip_ansi(text)
    lines = plain.splitlines() or [""]
    lh = line_height or int(font_size * 1.2)

    # Initialize PDF object writer
    writer = _PDFObjectWriter()

    # 1: Catalog (will point to Pages object 2 0 R)
    writer.add_object("1 0 obj\n<< /Type /Catalog /Pages 2 0 R >>\nendobj\n")

    # Placeholder for Pages; we need Kids references, so prepare list
    page_object_ids: list[int] = []

    # 3: Font (Courier)
    font_obj_id = writer.add_object("3 0 obj\n<< /Type /Font /Subtype /Type1 /BaseFont /Courier >>\nendobj\n")

    # Build pages and content streams
    current_y = page_height - top_margin
    content_lines: list[str] = ["BT", f"/F1 {font_size} Tf"]

    def flush_page(content: list[str]) -> None:
        # Close text object
        if content and content[-1] != "ET":
            content.append("ET")
        # Replace characters not encodable in Latin-1 (Courier built-in)
        stream_bytes = "\n".join(content).encode("latin-1", errors="replace")

        # Get next object IDs
        content_obj_id = writer._next_id
        page_obj_id = writer._next_id + 1

        # Create content stream object
        writer.add_object(
            f"{content_obj_id} 0 obj\n<< /Length {len(stream_bytes)} >>\nstream\n"
            + stream_bytes.decode("latin-1")
            + "\nendstream\nendobj\n"
        )
        # Create page object that references this stream
        writer.add_object(
            f"{page_obj_id} 0 obj\n<< /Type /Page /Parent 2 0 R /MediaBox [0 0 {page_width} {page_height}]\n"
            f"/Resources << /Font << /F1 {font_obj_id} 0 R >> >> /Contents {content_obj_id} 0 R >>\nendobj\n"
        )
        page_object_ids.append(page_obj_id)

    for line in lines:
        current_y -= lh
        if current_y < top_margin:
            # Emit current page and start a new one
            flush_page(content_lines)
            current_y = page_height - top_margin - lh
            content_lines = ["BT", f"/F1 {font_size} Tf"]
        content_lines.append(f"{left_margin} {current_y} Td ({_escape_pdf_text(line)}) Tj")

    # Flush the last page
    flush_page(content_lines)

    # 2: Pages object with Kids = all page ids
    kids = " ".join(f"{pid} 0 R" for pid in page_object_ids)
    writer.add_object(f"2 0 obj\n<< /Type /Pages /Kids [{kids}] /Count {len(page_object_ids)} >>\nendobj\n")

    # Compose final PDF
    objects = writer.get_objects()
    xref_positions = writer.get_xref_positions()
    body = b"%PDF-1.4\n" + b"".join(objects)
    xref_start = len(body)
    xref_header = f"xref\n0 {len(xref_positions) + 1}\n0000000000 65535 f \n"
    offset = len("%PDF-1.4\n".encode("latin-1"))
    xref_entries = []
    for pos in xref_positions:
        xref_entries.append(f"{pos + offset:010} 00000 n \n")
    trailer = (
        f"trailer\n<< /Size {len(xref_positions) + 1} /Root 1 0 R >>\nstartxref\n{xref_start}\n%%EOF\n"
    )
    pdf = body + xref_header.encode("latin-1") + "".join(xref_entries).encode("latin-1") + trailer.encode("latin-1")
    return pdf
