"""ANSI/markup to HTML exporter.

Hardened converter that parses ANSI SGR sequences and emits well-nested
``<span>`` elements with inline CSS for foreground/background colours and
text effects. Handles nested style transitions, partial resets, and 24-bit
(`38;2`/`48;2`) as well as 4-bit/8-bit colour codes. Newlines are preserved
via ``<br>`` and the whole block is wrapped in a styled ``<pre>``.
"""

from __future__ import annotations

import html
import re
from typing import Any

_SGR_RE = re.compile(r"\x1b\[([0-9;]*)m")

_FG_MAP: dict[int, str] = {
    # Standard
    30: "#000000",
    31: "#ef4444",
    32: "#22c55e",
    33: "#f59e0b",
    34: "#3b82f6",
    35: "#a855f7",
    36: "#06b6d4",
    37: "#e5e7eb",
    # Bright
    90: "#6b7280",
    91: "#f87171",
    92: "#34d399",
    93: "#fde047",
    94: "#60a5fa",
    95: "#c084fc",
    96: "#22d3ee",
    97: "#f3f4f6",
}

_BG_MAP: dict[int, str] = {
    # Standard
    40: "#000000",
    41: "#ef4444",
    42: "#22c55e",
    43: "#f59e0b",
    44: "#3b82f6",
    45: "#a855f7",
    46: "#06b6d4",
    47: "#e5e7eb",
    # Bright
    100: "#6b7280",
    101: "#f87171",
    102: "#34d399",
    103: "#fde047",
    104: "#60a5fa",
    105: "#c084fc",
    106: "#22d3ee",
    107: "#f3f4f6",
}


def _style_to_css(state: dict[str, Any]) -> str:
    parts: list[str] = []
    if state.get("bold"):
        parts.append("font-weight:bold")
    if state.get("dim"):
        parts.append("opacity:0.75")
    if state.get("italic"):
        parts.append("font-style:italic")
    td: list[str] = []
    if state.get("underline"):
        td.append("underline")
    if state.get("strike"):
        td.append("line-through")
    if td:
        parts.append(f"text-decoration:{' '.join(td)}")
    fg = state.get("fg")
    if fg:
        parts.append(f"color:{fg}")
    bg = state.get("bg")
    if bg:
        parts.append(f"background-color:{bg}")
    return ";".join(parts)


def _parse_and_render_html(text: str) -> str:
    out: list[str] = []
    state: dict[str, Any] = {}
    open_span = False
    pos = 0
    for m in _SGR_RE.finditer(text):
        # Emit preceding text
        if m.start() > pos:
            # Escape text content to prevent HTML injection
            out.append(html.escape(text[pos:m.start()], quote=True))

        # Apply SGR parameters
        params = m.group(1)
        if params == "" or params == "0":
            # Full reset
            if open_span:
                out.append("</span>")
                open_span = False
            state.clear()
        else:
            for p in params.split(";"):
                if not p:
                    continue
                code = int(p)
                if code == 0:
                    if open_span:
                        out.append("</span>")
                        open_span = False
                    state.clear()
                elif code == 1:
                    state["bold"] = True
                elif code == 2:
                    state["dim"] = True
                elif code == 3:
                    state["italic"] = True
                elif code == 4:
                    state["underline"] = True
                elif code == 9:
                    state["strike"] = True
                elif code == 22:
                    state.pop("bold", None)
                    state.pop("dim", None)
                elif code == 23:
                    state.pop("italic", None)
                elif code == 24:
                    state.pop("underline", None)
                elif code == 29:
                    state.pop("strike", None)
                elif code == 39:
                    state.pop("fg", None)
                elif code == 49:
                    state.pop("bg", None)
                elif code in _FG_MAP:
                    state["fg"] = _FG_MAP[code]
                elif code in _BG_MAP:
                    state["bg"] = _BG_MAP[code]
                else:
                    # Handle 24-bit colour sequences like 38;2;r;g;b / 48;2;r;g;b
                    # We need to look ahead into the remainder of the params
                    # Example: "38;2;255;0;0" or "48;2;1;2;3"
                    # We parse starting at this code if it is 38 or 48
                    if code in (38, 48):
                        # Attempt to parse "2;r;g;b" following
                        rest = params.split(";")
                        # Find the index of our current code in the sequence
                        try:
                            idx = rest.index(str(code))
                        except ValueError:
                            idx = -1
                        if idx != -1 and idx + 4 < len(rest) and rest[idx + 1] == "2":
                            try:
                                r = int(rest[idx + 2])
                                g = int(rest[idx + 3])
                                b = int(rest[idx + 4])
                                hexcol = f"rgb({r},{g},{b})"
                                if code == 38:
                                    state["fg"] = hexcol
                                else:
                                    state["bg"] = hexcol
                            except Exception:
                                pass
        # After applying, (re)open span for current state
        css = _style_to_css(state)
        if open_span:
            out.append("</span>")
            open_span = False
        if css:
            out.append(f'<span style="{css}">')
            open_span = True
        pos = m.end()

    # Trailing text
    if pos < len(text):
        out.append(html.escape(text[pos:], quote=True))
    if open_span:
        out.append("</span>")
    # Preserve newlines
    return "".join(out).replace("\n", "<br>\n")


def ansi_to_html(text: str) -> str:
    inner = _parse_and_render_html(text)
    return (
        "<pre style=\"background:#111;color:#e5e7eb;padding:12px;border-radius:6px;\">"
        + inner
        + "</pre>"
    )
