"""Declarative style transitions for text using Console.Scheduler.

Provides a Transition API to animate between two color specs or gradients.
"""

from __future__ import annotations

import time
from typing import TYPE_CHECKING

from ..core import Console
from ..style.colors import Color

if TYPE_CHECKING:
    from collections.abc import Callable


def _lerp(a: float, b: float, t: float) -> float:
    return a + (b - a) * t


def ease_in_out(t: float) -> float:
    # smoothstep-like
    return t * t * (3 - 2 * t)


class Transition:
    @staticmethod
    def color_fade(text: str, start: str, end: str, duration: float = 1.0, fps: int = 30, easing: Callable[[float], float] | None = None, console: Console | None = None) -> None:
        """Animate text color from start spec to end spec over duration."""
        easing = easing or ease_in_out
        c = console or Console.for_stdout()
        fps = max(1, int(fps))
        total = max(1, int(duration * fps))
        start_rgb = Color.resolve_rgb(start) or (255, 255, 255)
        end_rgb = Color.resolve_rgb(end) or (255, 255, 255)

        for i in range(total):
            t = easing(i / max(1, total - 1))
            r = int(_lerp(start_rgb[0], end_rgb[0], t))
            g = int(_lerp(start_rgb[1], end_rgb[1], t))
            b = int(_lerp(start_rgb[2], end_rgb[2], t))
            col = f"\033[38;2;{r};{g};{b}m"
            c.print(col + text + Color.RESET, end="\r", markup=False)
            time.sleep(1.0 / fps)
        c.print("")

    @staticmethod
    def blink(text: str, color: str = "fg", cycles: int = 3, fps: int = 6, console: Console | None = None) -> None:
        c = console or Console.for_stdout()
        col = Color.get_color(color)
        for _ in range(cycles):
            c.print(col + text + Color.RESET, end="\r", markup=False)
            time.sleep(1.0 / max(1, fps))
            c.print(" " * len(text), end="\r")
            time.sleep(1.0 / max(1, fps))
        c.print(col + text + Color.RESET)
