"""Animation effects extracted from the legacy implementation."""
from __future__ import annotations

import time

from ..style.colors import Color


class Animation:
    """Text animation effects."""

    @staticmethod
    def typewriter(
        text: str,
        delay: float = 0.05,
        color: str | None = None
    ) -> None:
        """
        Print text with typewriter effect.

        Args:
            text: Text to print
            delay: Delay between characters (seconds)
            color: Text color
        """
        text_color = Color.get_color(color) if color else ""
        if text_color:
            print(text_color, end="")

        for char in text:
            print(char, end="", flush=True)
            time.sleep(delay)

        if text_color:
            print(Color.RESET)
        else:
            print()

    @staticmethod
    def typewriter_iter(text: str, *, color: str | None = None):
        """Yield successive frames for a typewriter effect without sleeping."""
        text_color = Color.get_color(color) if color else ""
        prefix = text_color
        for i in range(1, len(text) + 1):
            yield prefix + text[:i] + (Color.RESET if color else "")

    @staticmethod
    def fade_in(
        text: str,
        steps: int = 5,
        delay: float = 0.1
    ) -> None:
        """
        Print text with fade-in effect (using dim to normal).

        Args:
            text: Text to print
            steps: Number of fade steps
            delay: Delay between steps (seconds)
        """
        for i in range(steps):
            # Clear line and print progressively less dim
            print("\r" + " " * len(text), end="\r")
            if i < steps - 1:
                print(Color.DIM + text + Color.RESET, end="", flush=True)
            else:
                print(text, end="", flush=True)
            time.sleep(delay)
        print()

    @staticmethod
    def fade_in_iter(text: str, *, steps: int = 5):
        for i in range(steps):
            if i < steps - 1:
                yield Color.DIM + text + Color.RESET
            else:
                yield text

    @staticmethod
    def pulse(text: str, cycles: int = 3, delay: float = 0.1, color: str = "fg") -> None:
        col = Color.get_color(color)
        for _ in range(cycles):
            print(f"\r{Color.DIM}{col}{text}{Color.RESET}", end="", flush=True)
            time.sleep(delay)
            print(f"\r{col}{text}{Color.RESET}", end="", flush=True)
            time.sleep(delay)
        print()

    @staticmethod
    def wave(text: str, cycles: int = 1, delay: float = 0.05, color: str = "fg") -> None:
        col = Color.get_color(color)
        import math
        for t in range(int(len(text) * 4 * cycles)):
            out = []
            for i, ch in enumerate(text):
                y = int((math.sin((i + t) / 2.0) + 1) * 1)
                out.append(" " * y + ch)
            print("\r" + col + " ".join(out) + Color.RESET, end="", flush=True)
            time.sleep(delay)
        print()

    @staticmethod
    def wave_iter(text: str, *, cycles: int = 1, color: str = "fg"):
        import math
        col = Color.get_color(color)
        for t in range(int(len(text) * 4 * cycles)):
            out = []
            for i, ch in enumerate(text):
                y = int((math.sin((i + t) / 2.0) + 1) * 1)
                out.append(" " * y + ch)
            yield col + " ".join(out) + Color.RESET

    @staticmethod
    def shake(text: str, cycles: int = 8, delay: float = 0.04, color: str = "fg") -> None:
        import random
        col = Color.get_color(color)
        for _ in range(cycles):
            offset = " " * random.randint(0, 3)
            print("\r" + offset + col + text + Color.RESET + "\x1b[0K", end="", flush=True)
            time.sleep(delay)
        print("\r" + col + text + Color.RESET)

    @staticmethod
    def gradient_pulse(
        text: str,
        palette: list[tuple[int, int, int]] | None = None,
        duration: float = 2.0,
        cycles: int = 1,
        fps: int = 24,
    ) -> None:
        """Animate a flowing gradient across text."""

        if not text:
            return

        palette = palette or [(0, 255, 255), (255, 0, 255), (255, 255, 0)]
        fps = max(1, fps)
        frame_delay = 1.0 / fps
        total_frames = max(1, int(duration * fps))

        def _blend(colors: list[tuple[int, int, int]], position: float) -> tuple[int, int, int]:
            if len(colors) == 1:
                return colors[0]
            position = position % 1.0
            segment = position * (len(colors) - 1)
            idx = int(segment)
            frac = segment - idx
            start = colors[idx]
            end = colors[min(idx + 1, len(colors) - 1)]
            r = int(start[0] + (end[0] - start[0]) * frac)
            g = int(start[1] + (end[1] - start[1]) * frac)
            b = int(start[2] + (end[2] - start[2]) * frac)
            return (r, g, b)

        for _ in range(max(1, cycles)):
            for frame in range(total_frames):
                phase = frame / total_frames
                start_col = _blend(palette, phase)
                end_col = _blend(palette, phase + 0.5)
                gradient_text = Color.gradient(text, start_col, end_col)
                print("\r" + gradient_text + Color.RESET, end="", flush=True)
                time.sleep(frame_delay)
        print()

    @staticmethod
    def particle_trail(
        text: str,
        duration: float = 2.0,
        fps: int = 30,
        color: str = "cyan",
        glyph: str = "•",
        trail_length: int = 8,
    ) -> None:
        """Emit a simple particle trail following the provided text."""
        if not text:
            return

        import random

        fps = max(1, fps)
        frame_delay = 1.0 / fps
        trail_length = max(1, trail_length)
        col = Color.get_color(color)
        particles: list[tuple[int, int]] = []

        for _ in range(int(duration * fps)):
            particles = [
                (position, ttl - 1)
                for (position, ttl) in particles
                if ttl > 1
            ]
            particles.append((random.randint(0, len(text)), trail_length))

            overlay = [" "] * (len(text) + trail_length)
            for pos, _ttl in particles:
                idx = min(len(overlay) - 1, pos)
                overlay[idx] = glyph

            overlay_text = "".join(overlay).rstrip()
            print(
                "\r"
                + col
                + text
                + Color.RESET
                + " "
                + Color.DIM
                + overlay_text
                + Color.RESET,
                end="",
                flush=True,
            )
            time.sleep(frame_delay)
        print()
