"""A tiny scripting DSL to invoke components by name.

Capabilities:
- Quoted args with spaces: Title text="Hello World" width=60
- Booleans: Checkbox checked=true
- Arrays/objects: ListElement items=["A", "B C"]; Meter thresholds={0.0:"green",0.6:"yellow"}
- Nested structures supported via Python literals or JSON

Example:
  ProgressBar current=72 maximum=100 label="Loading" color="green"
  Title text="Hello" width=60 color="primary"
"""

from __future__ import annotations

import ast
import json
from dataclasses import dataclass
from typing import Any

from . import components
from .core import render_call, tfprint


@dataclass(slots=True)
class _ParseError(Exception):
    message: str
    column: int | None = None

    def __str__(self) -> str:
        if self.column is None:
            return self.message
        return f"{self.message} (at column {self.column})"


def _tokenize_args(s: str) -> list[str]:
    """Split a key=value argument string into tokens while respecting quotes/brackets."""
    tokens: list[str] = []
    buf: list[str] = []
    quote: str | None = None
    depth = 0
    i = 0
    while i < len(s):
        ch = s[i]
        if quote is not None:
            buf.append(ch)
            if ch == quote:
                quote = None
            i += 1
            continue
        if ch in ('"', "'"):
            quote = ch
            buf.append(ch)
            i += 1
            continue
        if ch in "([{":
            depth += 1
            buf.append(ch)
            i += 1
            continue
        if ch in ")]}":
            depth = max(0, depth - 1)
            buf.append(ch)
            i += 1
            continue
        if ch.isspace() and depth == 0:
            if buf:
                tokens.append("".join(buf))
                buf = []
            # skip consecutive spaces
            while i < len(s) and s[i].isspace():
                i += 1
            continue
        buf.append(ch)
        i += 1
    if quote is not None:
        raise _ParseError("Unterminated string literal", column=len(s))
    if depth != 0:
        raise _ParseError("Unbalanced brackets in argument list", column=len(s))
    if buf:
        tokens.append("".join(buf))
    return tokens


def _parse_value(raw: str) -> object:
    v = raw.strip()
    # Fast path for booleans/null
    low = v.lower()
    if low == "true":
        return True
    if low == "false":
        return False
    if low in {"none", "null"}:
        return None
    # Try JSON for bracketed values
    if v[:1] in "[{":
        try:
            return json.loads(v)
        except Exception:
            pass
    # Try Python literal eval
    try:
        return ast.literal_eval(v)
    except Exception:
        # Fallback to number coercions
        try:
            if v.isdigit() or (v.startswith("-") and v[1:].isdigit()):
                return int(v)
            return float(v)
        except Exception:
            return v


def _resolve_component(name: str) -> Any:
    """Resolve a component render function by name.

    Accepts either the Class name (e.g., Title) or a snake_case function alias (e.g., title).
    Returns the underlying .render callable suitable for wrapping via render_call.
    """
    obj = getattr(components, name, None)
    if obj is not None and hasattr(obj, "render"):
        try:
            render_attr = obj.render
        except Exception:
            render_attr = None
        if callable(render_attr):
            return render_attr
    # Try snake_case helper mapping from CamelCase
    snake = []
    for c in name:
        if c.isupper() and snake:
            snake.append("_")
        snake.append(c.lower())
    alias = "".join(snake)
    func: Any = getattr(components, alias, None)  # Any: dynamically looked up component factories
    if callable(func):
        # func may be a factory returning a renderable; we wrap a small shim that invokes it via tfprint
        def _shim(**kwargs: object) -> None:
            factory: Any = func(**kwargs)  # Any: component factories return various types
            # If the factory returns a callable, attempt to call to get the actual renderable
            renderable: Any  # Any: renderables can be various types
            if callable(factory):
                try:
                    renderable = factory()
                except TypeError:
                    renderable = factory
            else:
                renderable = factory
            tfprint(renderable)
        return _shim
    raise ValueError(f"Unknown component: {name}")


def run_line(line: str) -> None:
    line = line.strip()
    if not line or line.startswith("#"):
        return
    # Split name and the rest
    first_space = line.find(" ")
    if first_space == -1:
        name = line
        arg_str = ""
    else:
        name = line[:first_space]
        arg_str = line[first_space + 1 :]
    # Resolve component render function
    try:
        render_func = _resolve_component(name)
    except Exception as e:
        raise ValueError(str(e)) from e
    # Parse key=value args
    kwargs: dict[str, object] = {}
    if arg_str:
        try:
            tokens = _tokenize_args(arg_str)
        except _ParseError as pe:
            raise ValueError(f"Parse error: {pe}") from pe
        for token in tokens:
            if "=" not in token:
                continue
            k, v = token.split("=", 1)
            kwargs[k] = _parse_value(v)
    # Render via render_call for composability
    tfprint(render_call(render_func, **kwargs))


def run(text: str) -> None:
    for line in text.splitlines():
        run_line(line)
