"""Minimal virtual DOM for text frames with line-diff patching.

This module provides simple data structures and utilities to compute and
apply structured changes between two line-based text frames. The intent is to
enable diff-friendly live rendering where only modified lines are rewritten.
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from collections.abc import Iterable


@dataclass(slots=True)
class Node:
    """A virtual node representing a text frame as a list of lines."""

    lines: list[str]


@dataclass(slots=True)
class LineChange:
    """A single line replacement at a given index.

    The absence of an index in the final "new" frame implies deletion of
    trailing lines (handled by consumers like LiveSession via line wipes).
    """

    index: int
    text: str


def diff_lines(old: list[str], new: list[str]) -> list[tuple[int, str]]:
    """Return a list of ``(index, new_line)`` changes to transform ``old`` -> ``new``.

    Deprecated in favor of :func:`diff_changes`, but retained for compatibility.
    """
    changes: list[tuple[int, str]] = []
    max_len = max(len(old), len(new))
    for i in range(max_len):
        a = old[i] if i < len(old) else ""
        b = new[i] if i < len(new) else ""
        if a != b:
            changes.append((i, b))
    return changes


def diff_changes(old: list[str], new: list[str]) -> list[LineChange]:
    """Compute structured line changes from ``old`` to ``new``.

    Returns a sorted list of :class:`LineChange` where each change is a full
    replacement of the line at ``index`` with ``text`` (which can be an empty
    string indicating a blank line). Trailing deletions are represented by the
    absence of indices >= ``len(new)``.
    """
    result: list[LineChange] = []
    max_len = max(len(old), len(new))
    for i in range(max_len):
        a = old[i] if i < len(old) else ""
        b = new[i] if i < len(new) else ""
        if a != b:
            result.append(LineChange(index=i, text=b))
    return result


def apply_changes(old: list[str], changes: Iterable[LineChange]) -> list[str]:
    """Apply a set of line changes to ``old`` and return the resulting frame.

    This does not implicitly truncate; callers should slice the result to the
    intended new length if representing frame shrinkage.
    """
    lines = list(old)
    for change in changes:
        idx = change.index
        # Extend with blanks if necessary
        if idx >= len(lines):
            lines.extend([""] * (idx + 1 - len(lines)))
        lines[idx] = change.text
    return lines
