"""Minimal Win32 native window for custom rendering without external deps.

This module provides a very small wrapper around the Win32 API using ctypes to
create a window, run a message loop on a dedicated thread, and render a text
frame with ANSI-to-style translation via the translator module.
"""
from __future__ import annotations

import ctypes
import ctypes.wintypes as wt
import threading
from typing import TYPE_CHECKING

from ...utils.logging import get_logger
from .translator import parse_line_to_spans

if TYPE_CHECKING:
    from collections.abc import Callable

LRESULT = wt.LPARAM
WNDPROC = ctypes.WINFUNCTYPE(LRESULT, wt.HWND, wt.UINT, wt.WPARAM, wt.LPARAM)

CS_HREDRAW = 0x0002
CS_VREDRAW = 0x0001

WM_DESTROY = 0x0002
WM_PAINT = 0x000F
WM_KEYDOWN = 0x0100

SW_SHOW = 5

WHITE_BRUSH = 0

DT_NOCLIP = 0x0100

user32 = ctypes.windll.user32
gdi32 = ctypes.windll.gdi32
kernel32 = ctypes.windll.kernel32


class Win32Window:
    def __init__(
        self,
        title: str = "Textforge Window",
        width: int = 800,
        height: int = 600,
        on_key: Callable[[str], None] | None = None,
    ) -> None:
        self.title = title
        self.width = width
        self.height = height
        self._hwnd: wt.HWND | None = None
        self._thread: threading.Thread | None = None
        self._running = False
        self._frame_text: str = ""
        self._lock = threading.Lock()
        self._on_key = on_key
        self._ready = threading.Event()
        self._error: Exception | None = None
        self._register_and_start()

    def _register_and_start(self) -> None:
        def run() -> None:
            try:
                class WNDCLASS(ctypes.Structure):
                    _fields_ = [
                        ("style", wt.UINT),
                        ("lpfnWndProc", WNDPROC),
                        ("cbClsExtra", wt.LPARAM),
                        ("cbWndExtra", wt.LPARAM),
                        ("hInstance", wt.HINSTANCE),
                        ("hIcon", wt.HICON),
                        ("hCursor", wt.HCURSOR),
                        ("hbrBackground", wt.HBRUSH),
                        ("lpszMenuName", wt.LPCWSTR),
                        ("lpszClassName", wt.LPCWSTR),
                    ]

                # Reset last error so we can detect a failure when RegisterClassW returns 0.
                if hasattr(ctypes, "set_last_error"):
                    ctypes.set_last_error(0)
                wc = WNDCLASS()
                hinst = kernel32.GetModuleHandleW(None)
                wc.style = CS_HREDRAW | CS_VREDRAW
                wc.lpfnWndProc = WNDPROC(self._wndproc)
                wc.hInstance = hinst
                wc.hCursor = user32.LoadCursorW(None, 32512)
                wc.hbrBackground = gdi32.GetStockObject(WHITE_BRUSH)
                wc.lpszClassName = "TextforgeWndClass"
                atom = user32.RegisterClassW(ctypes.byref(wc))
                if atom == 0:
                    err = ctypes.get_last_error() if hasattr(ctypes, "get_last_error") else 0
                    # ERROR_CLASS_ALREADY_EXISTS (1410) is expected when re-running.
                    if err not in (0, 1410):
                        raise ctypes.WinError(err)

                hwnd = user32.CreateWindowExW(
                    0,
                    wc.lpszClassName,
                    self.title,
                    0xCF0000,  # WS_OVERLAPPEDWINDOW
                    100,
                    100,
                    self.width,
                    self.height,
                    None,
                    None,
                    hinst,
                    None,
                )
                if not hwnd:
                    err = ctypes.get_last_error() if hasattr(ctypes, "get_last_error") else 0
                    raise ctypes.WinError(err or 0)

                self._hwnd = hwnd
                user32.ShowWindow(hwnd, SW_SHOW)
                user32.UpdateWindow(hwnd)
                self._running = True
                self._ready.set()

                msg = wt.MSG()
                while user32.GetMessageW(ctypes.byref(msg), None, 0, 0) != 0:
                    user32.TranslateMessage(ctypes.byref(msg))
                    user32.DispatchMessageW(ctypes.byref(msg))
            except Exception as exc:
                self._error = exc
                get_logger().debug(f"Win32 window thread failed: {exc}")
                self._running = False
                self._ready.set()
            finally:
                self._running = False
                if not self._ready.is_set():
                    self._ready.set()

        self._thread = threading.Thread(target=run, daemon=True)
        self._thread.start()

    def _wndproc(self, hwnd: wt.HWND, msg: int, wparam: wt.WPARAM, lparam: wt.LPARAM) -> int:
        if msg == WM_PAINT:
            class PAINTSTRUCT(ctypes.Structure):
                _fields_ = [
                    ("hdc", wt.HDC),
                    ("fErase", wt.BOOL),
                    ("rcPaint", wt.RECT),
                    ("fRestore", wt.BOOL),
                    ("fIncUpdate", wt.BOOL),
                    ("rgbReserved", wt.BYTE * 32),
                ]

            ps = PAINTSTRUCT()
            hdc = user32.BeginPaint(hwnd, ctypes.byref(ps))
            # Create monospace font
            hfont = gdi32.CreateFontW(
                18,
                0,
                0,
                0,
                400,
                False,
                False,
                False,
                0,
                0,
                0,
                0,
                0,
                "Consolas",
            )
            old = gdi32.SelectObject(hdc, hfont)
            try:
                with self._lock:
                    frame = self._frame_text
                y = 10
                for line in frame.split("\n"):
                    x = 10
                    for span in parse_line_to_spans(line):
                        # parse css color string to COLORREF
                        r, g, b = _parse_css_color(span.fg) if span.fg else (0, 0, 0)
                        colorref = (b << 16) | (g << 8) | r
                        gdi32.SetTextColor(hdc, colorref)
                        # Draw text span
                        text = span.text
                        gdi32.TextOutW(hdc, x, y, text, len(text))
                        x += _approx_text_width(text)
                    y += 20
            finally:
                gdi32.SelectObject(hdc, old)
                gdi32.DeleteObject(hfont)
                user32.EndPaint(hwnd, ctypes.byref(ps))
            return 0
        if msg == WM_KEYDOWN:
            callback = self._on_key
            if callback is not None:
                token = _translate_virtual_key(int(wparam))
                if token:
                    try:
                        callback(token)
                    except Exception as e:
                        get_logger().warning(f"Key callback failed: {e}")
            return 0
        if msg == WM_DESTROY:
            user32.PostQuitMessage(0)
            return 0
        return user32.DefWindowProcW(hwnd, msg, wparam, lparam)

    def wait_until_ready(self, timeout: float | None = 2.0) -> bool:
        self._ready.wait(timeout=timeout)
        if self._error is not None:
            get_logger().debug(f"Win32 window initialization failed: {self._error}")
        return self._hwnd is not None and self._error is None

    def update_text(self, text: str) -> None:
        with self._lock:
            self._frame_text = text
        if self._hwnd:
            user32.InvalidateRect(self._hwnd, None, True)

    def stop(self) -> None:
        hwnd = self._hwnd
        if hwnd:
            user32.PostMessageW(hwnd, WM_DESTROY, 0, 0)
        t = self._thread
        if t and t.is_alive():
            t.join(timeout=0.5)
        self._hwnd = None
        self._running = False


def _approx_text_width(text: str) -> int:
    # Approx monospace width: 10px per glyph
    return len(text) * 10


def _parse_css_color(css: str) -> tuple[int, int, int]:
    try:
        if not css:
            return (0, 0, 0)
        if css.startswith("#") and len(css) == 7:
            r = int(css[1:3], 16)
            g = int(css[3:5], 16)
            b = int(css[5:7], 16)
            return (r, g, b)
        if css.startswith("rgb(") and css.endswith(")"):
            parts = css[4:-1].split(",")
            r = int(parts[0])
            g = int(parts[1])
            b = int(parts[2])
            return (r, g, b)
    except Exception:
        return (0, 0, 0)
    return (0, 0, 0)


def _translate_virtual_key(value: int) -> str:
    mapping = {
        0x08: "backspace",
        0x09: "tab",
        0x0D: "enter",
        0x1B: "escape",
        0x20: "space",
        0x21: "page_up",
        0x22: "page_down",
        0x23: "end",
        0x24: "home",
        0x25: "left",
        0x26: "up",
        0x27: "right",
        0x28: "down",
        0x70: "f1",
        0x71: "f2",
        0x72: "f3",
        0x73: "f4",
        0x74: "f5",
        0x75: "f6",
        0x76: "f7",
        0x77: "f8",
        0x78: "f9",
        0x79: "f10",
        0x7A: "f11",
        0x7B: "f12",
    }
    if value in mapping:
        return mapping[value]
    if 0x30 <= value <= 0x39:
        return chr(value)
    if 0x41 <= value <= 0x5A:
        return chr(value).lower()
    return ""
