"""Translate Textforge ANSI/markup to GUI spans.

We leverage existing export HTML/SVG color mapping logic to parse ANSI SGR and
produce span runs with fg/bg and style flags suitable for GUI drawing.
"""
from __future__ import annotations

import re
from dataclasses import dataclass
from typing import Any

_SGR_RE = re.compile(r"\x1b\[([0-9;]*)m")

_FG_MAP: dict[int, str] = {
    30: "#000000",
    31: "#ef4444",
    32: "#22c55e",
    33: "#f59e0b",
    34: "#3b82f6",
    35: "#a855f7",
    36: "#06b6d4",
    37: "#e5e7eb",
    90: "#6b7280",
    91: "#f87171",
    92: "#34d399",
    93: "#fde047",
    94: "#60a5fa",
    95: "#c084fc",
    96: "#22d3ee",
    97: "#f3f4f6",
}

_BG_MAP: dict[int, str] = {
    40: "#000000",
    41: "#ef4444",
    42: "#22c55e",
    43: "#f59e0b",
    44: "#3b82f6",
    45: "#a855f7",
    46: "#06b6d4",
    47: "#e5e7eb",
    100: "#6b7280",
    101: "#f87171",
    102: "#34d399",
    103: "#fde047",
    104: "#60a5fa",
    105: "#c084fc",
    106: "#22d3ee",
    107: "#f3f4f6",
}


@dataclass(slots=True)
class Span:
    text: str
    fg: str | None = None
    bg: str | None = None
    bold: bool = False
    italic: bool = False
    underline: bool = False
    strike: bool = False


def parse_line_to_spans(line: str) -> list[Span]:
    spans: list[Span] = []
    state: dict[str, Any] = {}
    buf: list[str] = []
    pos = 0
    for m in _SGR_RE.finditer(line):
        if m.start() > pos:
            buf.append(line[pos:m.start()])
        if buf:
            spans.append(_span_from_buf(buf, state))
            buf.clear()
        params = m.group(1)
        if params == "" or params == "0":
            state.clear()
        else:
            parts = params.split(";")
            i = 0
            while i < len(parts):
                p = parts[i]
                if not p:
                    i += 1
                    continue
                code = int(p)
                if code == 0:
                    state.clear()
                elif code == 1:
                    state["bold"] = True
                elif code == 3:
                    state["italic"] = True
                elif code == 4:
                    state["underline"] = True
                elif code == 9:
                    state["strike"] = True
                elif code == 22:
                    state.pop("bold", None)
                elif code == 23:
                    state.pop("italic", None)
                elif code == 24:
                    state.pop("underline", None)
                elif code == 29:
                    state.pop("strike", None)
                elif code in _FG_MAP:
                    state["fg"] = _FG_MAP[code]
                elif code in _BG_MAP:
                    state["bg"] = _BG_MAP[code]
                elif code in (38, 48) and i + 4 < len(parts) and parts[i + 1] == "2":
                    try:
                        r = int(parts[i + 2])
                        g = int(parts[i + 3])
                        b = int(parts[i + 4])
                        css = f"rgb({r},{g},{b})"
                        if code == 38:
                            state["fg"] = css
                        else:
                            state["bg"] = css
                    except Exception:
                        pass
                    i += 4
                elif code == 39:
                    state.pop("fg", None)
                elif code == 49:
                    state.pop("bg", None)
                i += 1
        pos = m.end()
    if pos < len(line):
        buf.append(line[pos:])
    if buf:
        spans.append(_span_from_buf(buf, state))
    return spans


def _span_from_buf(buf: list[str], state: dict[str, Any]) -> Span:
    text = "".join(buf)
    return Span(
        text=text,
        fg=state.get("fg"),
        bg=state.get("bg"),
        bold=bool(state.get("bold")),
        italic=bool(state.get("italic")),
        underline=bool(state.get("underline")),
        strike=bool(state.get("strike")),
    )
