"""Threading helpers for GUI: cross-thread event posting and worker pool.

Python 3.14 free-threading is supported; these utilities provide explicit
thread-safe queues and posting to the EventBus without assumptions.
"""
from __future__ import annotations

import queue
import threading
from dataclasses import dataclass
from typing import TYPE_CHECKING, Any

if TYPE_CHECKING:
    from collections.abc import Callable


@dataclass(slots=True)
class WorkItem:
    fn: Callable[[], Any]


class Worker:
    def __init__(self) -> None:
        self._q: queue.Queue[WorkItem] = queue.Queue()
        self._running = True
        self._t = threading.Thread(target=self._loop, daemon=True)
        self._t.start()

    def _loop(self) -> None:
        while self._running:
            try:
                item = self._q.get(timeout=0.1)
            except Exception:
                continue
            try:
                item.fn()
            except Exception:
                pass

    def post(self, fn: Callable[[], Any]) -> None:
        self._q.put(WorkItem(fn=fn))

    def stop(self) -> None:
        self._running = False
