"""Runtime glue between Console streams and the native GUI window.

This module centralises management of a singleton GUI runtime so that the
``Console`` backend can present Textforge frames without each caller having to
coordinate window lifecycle or message pumps.

The runtime exposes a lightweight ``GuiStream`` object that mimics ``TextIO``
and understands two main operations:

* regular ``write``/``flush`` calls append to the current frame (useful for
  drop-in ``print`` compatibility)
* ``present_frame`` replaces the entire frame, which is how ``Console.live``
  drives real-time updates

The implementation prefers the native Win32 window (see ``win32_window``) and
falls back to a headless collector that simply writes to ``stdout`` when a GUI
cannot be created (for example during CI runs).
"""

from __future__ import annotations

import io
import os
import queue
import sys
import threading
from typing import TextIO

from ...utils.logging import get_logger
from .gui_renderer import Surface

_RUNTIME_LOCK = threading.Lock()
_RUNTIME: GuiRuntime | None = None


def get_runtime(create: bool = True) -> GuiRuntime | None:
    """Return the process-wide GUI runtime, creating it on demand."""
    global _RUNTIME
    with _RUNTIME_LOCK:
        if _RUNTIME is None and create:
            _RUNTIME = GuiRuntime()
        return _RUNTIME


class GuiRuntime:
    """Owns the GUI surface, native window and shared key queue."""

    def __init__(self) -> None:
        self._lock = threading.Lock()
        self._frame: str = ""
        self.surface = Surface()
        self._key_queue: queue.Queue[str] | None = None
        self._refcount = 0
        self._window = self._create_native_window()
        self._fallback_stream: TextIO | None = None
        if self._window is None:
            self._fallback_stream = sys.stdout
        else:
            self._window.update_text("")

    # -- window creation helpers -------------------------------------------------
    def _create_native_window(self):
        # Honour an env flag so automated tests can disable window creation.
        if os.environ.get("TEXTFORGE_HEADLESS", "").lower() in {"1", "true", "yes"}:
            return None
        try:
            from .win32_window import Win32Window
        except Exception:
            return None

        key_queue: queue.Queue[str] = queue.Queue()

        def _on_key(token: str) -> None:
            if token:
                key_queue.put(token)

        try:
            window = Win32Window("Textforge GUI", 960, 640, on_key=_on_key)
            if not window.wait_until_ready():
                window.stop()
                get_logger().debug("GUI window failed to become ready, falling back to stdout")
                return None
        except Exception as e:
            get_logger().debug(f"GUI window creation failed: {e}, falling back to stdout")
            return None
        self._key_queue = key_queue
        return window

    # -- reference counting ------------------------------------------------------
    def create_stream(self) -> GuiStream:
        with self._lock:
            self._refcount += 1
        stream = GuiStream(self)
        # Attempt to create the window eagerly so the first write shows up immediately.
        self.ensure_window()
        return stream

    def release_stream(self) -> None:
        destroy = False
        with self._lock:
            self._refcount -= 1
            if self._refcount <= 0:
                destroy = True
        if destroy:
            self.shutdown()

    # -- presentation ------------------------------------------------------------
    def append_text(self, chunk: str) -> None:
        if not chunk:
            return
        with self._lock:
            self._frame += chunk
            frame = self._frame
        self.surface.set_frame(frame)
        self._send_to_window(frame, incremental=chunk)

    def present_frame(self, text: str) -> None:
        with self._lock:
            self._frame = text
        self.surface.set_frame(text)
        self._send_to_window(text, incremental=None)

    def clear(self) -> None:
        self.present_frame("")

    def _send_to_window(self, text: str, *, incremental: str | None) -> None:
        if self._window is None:
            self.ensure_window()
        if self._window is not None:
            try:
                self._window.update_text(text)
                return
            except Exception:
                # If the native window dies mid-run we fall back to stdout.
                self._window = None
        if self._fallback_stream is not None:
            try:
                payload = incremental if incremental is not None else text
                self._fallback_stream.write(payload)
                self._fallback_stream.flush()
            except Exception as e:
                get_logger().debug(f"Failed to write to fallback stream: {e}")

    # -- key handling ------------------------------------------------------------
    def get_key_queue(self) -> queue.Queue[str] | None:
        return self._key_queue

    def ensure_window(self) -> bool:
        if self._window is not None:
            return True
        new_window = self._create_native_window()
        if new_window is not None:
            self._window = new_window
            with self._lock:
                current_frame = self._frame
            try:
                new_window.update_text(current_frame)
            except Exception as e:
                get_logger().debug(f"Failed to update text on new window: {e}")
            return True
        return False

    # -- lifecycle ----------------------------------------------------------------
    def shutdown(self) -> None:
        with _RUNTIME_LOCK:
            global _RUNTIME
            if _RUNTIME is self:
                _RUNTIME = None
        if self._window is not None:
            try:
                self._window.stop()
            except Exception as e:
                get_logger().debug(f"Failed to stop window during shutdown: {e}")
            self._window = None
        # Clear frame so future get_runtime(False) returns a clean slate.
        with self._lock:
            self._frame = ""
            self.surface.set_frame("")
        self._key_queue = None


class GuiStream(io.TextIOBase):
    """Stream handed to ``Console`` when the GUI backend is selected."""

    def __init__(self, runtime: GuiRuntime) -> None:
        super().__init__()
        self._runtime = runtime
        self._buffer: list[str] = []
        self._closed = False

    # -- textio api --------------------------------------------------------------
    @property
    def encoding(self) -> str:
        return "utf-8"

    def writable(self) -> bool:
        return True

    def write(self, text: str) -> int:
        if self._closed:
            raise ValueError("I/O operation on closed GUI stream")
        self._buffer.append(text)
        return len(text)

    def flush(self) -> None:
        if self._closed:
            return
        if not self._buffer:
            return
        chunk = "".join(self._buffer)
        self._buffer.clear()
        self._runtime.append_text(chunk)

    # -- console live integration ------------------------------------------------
    def present_frame(self, text: str) -> None:
        if self._closed:
            return
        # Clear any pending buffered writes to avoid stale data when switching
        # from append-mode (print) to full-frame mode (live sessions).
        self._buffer.clear()
        self._runtime.present_frame(text)

    def clear_frame(self) -> None:
        if self._closed:
            return
        self._buffer.clear()
        self._runtime.clear()

    @property
    def surface(self) -> Surface:
        return self._runtime.surface

    def close(self) -> None:
        if self._closed:
            return
        try:
            self.flush()
        finally:
            self._runtime.release_stream()
            self._closed = True

    # -- Helpers used by tests and utilities ------------------------------------
    def get_key_queue(self) -> queue.Queue[str] | None:
        return self._runtime.get_key_queue()
