"""Core GUI renderer primitives: Application, Window, Surface, LiveSession.

This is a custom, dependency-free GUI framework skeleton with thread-safe
event dispatch and an optional OpenGL drawing surface abstraction.
"""
from __future__ import annotations

import threading
from typing import TYPE_CHECKING, TextIO

from ...utils.logging import get_logger
from ..events import EventBus, QuitEvent, TickEvent

if TYPE_CHECKING:
    from collections.abc import Callable


class LiveSession:
    """GUI LiveSession that delegates frame text to an in-memory surface."""

    def __init__(self, stream: TextIO) -> None:
        self._closed = False
        surface = getattr(stream, "surface", None)
        if isinstance(surface, Surface):
            self._surface = surface
        else:
            self._surface = Surface()
        presenter = getattr(stream, "present_frame", None)
        self._present: Callable[[str], object] | None = presenter if callable(presenter) else None
        clearer = getattr(stream, "clear_frame", None)
        self._clear: Callable[[], object] | None = clearer if callable(clearer) else None
        if self._present is not None:
            try:
                self._present(self._surface.get_frame())
            except Exception as e:
                get_logger().debug(f"Failed to present initial frame: {e}")

    def update(self, text: str) -> None:
        self._surface.set_frame(text)
        if self._present is not None:
            try:
                self._present(text)
            except Exception as e:
                get_logger().debug(f"Failed to present frame: {e}")

    def close(self) -> None:
        if self._clear is not None:
            try:
                self._clear()
            except Exception as e:
                get_logger().debug(f"Failed to clear frame: {e}")
        self._closed = True


class Surface:
    """String frame surface with simple dimension tracking."""

    def __init__(self) -> None:
        self._frame: str = ""
        self.width: int = 0
        self.height: int = 0

    def set_frame(self, text: str) -> None:
        self._frame = text
        lines = text.split("\n")
        self.height = len(lines)
        self.width = max((len(line) for line in lines), default=0)

    def get_frame(self) -> str:
        return self._frame


class Application:
    """Top-level GUI application with a custom event loop and window stack."""

    def __init__(self) -> None:
        self._bus = EventBus()
        self._windows: list[Window] = []
        self._running = False
        self._lock = threading.Lock()

    @property
    def events(self) -> EventBus:
        return self._bus

    def add_window(self, window: Window) -> None:
        with self._lock:
            window.attach(self._bus)
            self._windows.append(window)

    def remove_window(self, window: Window) -> None:
        with self._lock:
            try:
                self._windows.remove(window)
            except ValueError:
                pass
            window.detach()

    def run(self, *, fps: float = 30.0) -> None:
        import time as _t
        self._running = True
        last = _t.perf_counter()
        interval = 1.0 / max(1e-6, fps)
        while self._running and any(w.visible for w in self._windows):
            now = _t.perf_counter()
            dt = now - last
            last = now
            self._bus.publish("tick", TickEvent(dt=dt))
            with self._lock:
                for w in list(self._windows):
                    w.paint()
            sleep_for = interval - (_t.perf_counter() - now)
            if sleep_for > 0:
                _t.sleep(sleep_for)

    def quit(self) -> None:
        self._running = False
        self._bus.publish("quit", QuitEvent(reason="app.quit"))


class Window:
    """A simple window with a paint callback and visibility state."""

    def __init__(self, title: str, surface: Surface | None = None) -> None:
        self.title = title
        self.surface = surface or Surface()
        self.visible = True
        self._bus: EventBus | None = None
        self._subs: list[Callable[[], None]] = []
        self._on_paint: Callable[[Window], None] | None = None

    def attach(self, bus: EventBus) -> None:
        self._bus = bus
        self._subs.append(bus.subscribe("quit", self._on_quit))

    def detach(self) -> None:
        for u in self._subs:
            try:
                u()
            except Exception as e:
                get_logger().debug(f"Failed to unsubscribe event handler: {e}")
        self._subs.clear()
        self._bus = None

    def on_paint(self, callback: Callable[[Window], None]) -> None:
        self._on_paint = callback

    def paint(self) -> None:
        if self._on_paint is not None:
            try:
                self._on_paint(self)
            except Exception as e:
                get_logger().debug(f"Failed to execute paint callback: {e}")

    def _on_quit(self, _evt: QuitEvent) -> None:
        self.visible = False
