from __future__ import annotations

import io
import sys
import threading
from typing import Protocol, TextIO

from ...utils.input import read_key
from ...utils.logging import get_logger
from .session import LiveSession


class WindowLike(Protocol):
    """Protocol for window objects that TerminalStream can use."""

    def write(self, text: str) -> int:
        """Write text to the window."""
        ...

    def flush(self) -> None:
        """Flush any buffered output."""
        ...

    def stop(self) -> None:
        """Stop the window."""
        ...


class TerminalWindow:
    def __init__(self, stream: TextIO | None = None, *, fps: float = 30.0) -> None:
        self._stream = stream or sys.stdout
        self._live = LiveSession(self._stream)
        self._buffer_lock = threading.Lock()
        self._buffer: list[str] = []
        self._dirty = threading.Event()
        self._running = False
        self._thread = threading.Thread(target=self._loop, daemon=True)
        self._fps = fps
        self._should_close = threading.Event()
        self._thread.start()

    def _loop(self) -> None:
        self._running = True
        try:
            interval = 1.0 / max(1e-6, self._fps)
            while not self._should_close.is_set():
                # Poll for key to allow closing
                try:
                    key = read_key(timeout=0.0)
                except Exception:
                    key = ""
                if key in ("q", "escape"):
                    break
                # Render when dirty
                if self._dirty.wait(timeout=interval):
                    self._dirty.clear()
                    with self._buffer_lock:
                        text = "".join(self._buffer)
                    self._live.update(text)
            # final frame flush
            with self._buffer_lock:
                text = "".join(self._buffer)
            if text:
                self._live.update(text)
        finally:
            try:
                self._live.close()
            except Exception as e:
                get_logger().debug(f"Failed to close live session: {e}")
            self._running = False

    def write(self, text: str) -> int:
        with self._buffer_lock:
            self._buffer.append(text)
        self._dirty.set()
        return len(text)

    def flush(self) -> None:
        # Immediate render on flush
        self._dirty.set()

    def stop(self) -> None:
        self._should_close.set()
        self.join()

    def join(self, timeout: float | None = None) -> None:
        if self._thread.is_alive():
            self._thread.join(timeout=timeout)


class TerminalStream(io.TextIOBase):
    def __init__(self, window) -> None:
        super().__init__()
        self._window = window

    @property
    def encoding(self) -> str:
        return "utf-8"

    def write(self, s: str) -> int:
        return self._window.write(s)

    def flush(self) -> None:
        self._window.flush()

    def close(self) -> None:
        self._window.stop()
        super().close()
