from __future__ import annotations

import time as _t
from typing import TYPE_CHECKING, TextIO

from ...utils.input import read_key
from ..events import EventBus, KeyEvent, QuitEvent, TickEvent
from .mouse import disable_mouse_reporting, enable_mouse_reporting
from .session import LiveSession

if TYPE_CHECKING:
    from collections.abc import Callable


class TerminalSession(LiveSession):
    """Extended live session with input-driven app loop support."""

    def __init__(self, stream: TextIO) -> None:
        super().__init__(stream)
        self._bus = EventBus()
        self._running = False

    @property
    def events(self) -> EventBus:
        """Access the event bus for subscribing to input and timer events."""
        return self._bus

    def run(self, app: TerminalApp, *, fps: float = 30.0) -> None:
        """Run the application in a live terminal session with input handling."""
        self._running = True
        app.attach(self._bus)
        frame: str = app.render()
        # Enable mouse reporting in terminal
        try:
            self.update(enable_mouse_reporting() + frame)
        except Exception:
            self.update(frame)

        interval = 1.0 / max(1e-6, fps)
        last = _t.perf_counter()
        while self._running and not app.should_quit:
            now = _t.perf_counter()
            dt = now - last
            last = now
            self._bus.publish("tick", TickEvent(dt=dt))
            key = read_key(timeout=0.0)
            if key:
                self._bus.publish("key", KeyEvent(key=key))
            new_frame = app.render()
            if new_frame != frame:
                frame = new_frame
                self.update(frame)
            sleep_for = interval - (_t.perf_counter() - now)
            if sleep_for > 0:
                _t.sleep(sleep_for)
        self._bus.publish("quit", QuitEvent())
        app.detach()
        # Disable mouse reporting and close
        try:
            self.update(disable_mouse_reporting())
        except Exception:
            pass
        self.close()

    def stop(self) -> None:
        self._running = False


class TerminalApp:
    """Base class for custom terminal applications."""

    def __init__(self) -> None:
        self._bus: EventBus | None = None
        self.should_quit: bool = False
        self._subs: list[Callable[[], None]] = []

    def attach(self, bus: EventBus) -> None:
        self._bus = bus
        self._subs.append(bus.subscribe("key", self._on_key_event))
        self._subs.append(bus.subscribe("tick", self._on_tick_event))
        self._subs.append(bus.subscribe("quit", self._on_quit_event))

    def detach(self) -> None:
        for unsub in self._subs:
            try:
                unsub()
            except Exception:
                pass
        self._subs.clear()
        self._bus = None

    def _on_key_event(self, evt: KeyEvent) -> None:
        try:
            self.on_key(evt.key)
        except Exception:
            pass

    def _on_tick_event(self, evt: TickEvent) -> None:
        try:
            self.on_tick(evt.dt)
        except Exception:
            pass

    def _on_quit_event(self, _evt: QuitEvent) -> None:
        self.should_quit = True

    def on_key(self, key: str) -> None:
        if key in ("q", "escape", "ctrl+c"):
            self.should_quit = True

    def on_tick(self, dt: float) -> None:
        _ = dt

    def render(self) -> str:
        return ""
